<?php
require_once 'includes/bootstrap.php';

if (isset($_SESSION['user_id'])) {
    header('Location: /administracion/inicio.php');
    exit();
}

if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Acceso al Sistema | AMI - Asistencia Médica Inmediata</title>
    
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    
    <style>
        /* --- VARIABLES DE DISEÑO (TEMA OSCURO) --- */
        :root {
            --color-primary: #78E320;
            --color-primary-dark: #58B01F;
            --color-primary-light: #e0f6d2;
            --color-background-dark: #2C3E50;
            --color-card-background: #1A1A1A;
            --color-text-primary: #EAECEE;
            --color-text-secondary: #A9A9A9;
            --color-border: #444444;
            --color-input-focus: rgba(120, 227, 32, 0.2);
            --color-error: #E74C3C;
            --shadow-light: 0px 8px 20px rgba(0, 0, 0, 0.25);
            --font-family: 'Inter', sans-serif;
        }

        /* --- ESTILOS GENERALES Y FONDO ANIMADO --- */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: var(--font-family);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            background-color: var(--color-background-dark);
            color: var(--color-text-primary);
            overflow: hidden;
            position: relative;
        }

        #background-canvas {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
            background-color: var(--color-background-dark);
        }

        /* --- TARJETA DE LOGIN --- */
        .login-container {
            position: relative;
            z-index: 10;
            width: 400px; /* Ancho ajustado para el contenido interno */
            max-width: 90%;
            padding: 40px;
            background-color: var(--color-card-background);
            border-radius: 12px;
            box-shadow: var(--shadow-light);
            text-align: center;
        }

        /* --- CABECERA DE LA TARJETA (LOGO Y TEXTO) --- */
        .login-header {
            margin-bottom: 30px;
        }

        .login-header .logo-img {
            max-width: 150px; /* Tamaño ajustado del logo */
            height: auto;
            display: block;
            margin: 0 auto 25px auto; /* Centrado y con margen inferior */
            /* --- CAMBIO: Filtro para hacer el logo blanco --- */
            filter: brightness(0) invert(1); 
        }

        .login-header h2 {
            font-size: 24px;
            font-weight: 700;
            color: var(--color-text-primary);
            margin-bottom: 8px;
        }

        .login-header p {
            font-size: 15px;
            color: var(--color-text-secondary);
            line-height: 1.5;
        }
        
        /* --- ESTILOS DEL FORMULARIO --- */
        .input-group {
            margin-bottom: 20px;
            text-align: left;
        }

        .input-group label {
            display: block;
            font-weight: 500;
            margin-bottom: 8px;
            font-size: 14px;
            color: var(--color-text-primary);
        }
        
        .input-wrapper {
            position: relative;
        }

        .input-wrapper .input-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--color-text-secondary);
            font-size: 18px;
            transition: color 0.3s ease;
        }

        .input-wrapper input {
            width: 100%;
            padding: 14px 15px 14px 50px;
            border: 1px solid var(--color-border);
            border-radius: 8px;
            font-size: 16px;
            font-family: var(--font-family);
            background-color: #2D2D2D;
            color: var(--color-text-primary);
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }

        .input-wrapper input::placeholder {
            color: var(--color-text-secondary);
            opacity: 0.7;
        }

        .input-wrapper input:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px var(--color-input-focus);
        }

        .input-wrapper input:focus ~ .input-icon {
            color: var(--color-primary);
        }
        
        .error-message {
            color: var(--color-error);
            background-color: rgba(231, 76, 60, 0.1);
            border: 1px solid rgba(231, 76, 60, 0.2);
            padding: 10px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
            min-height: 45px;
            display: flex;
            align-items: center;
            justify-content: center;
            text-align: center;
        }

        /* --- BOTÓN DE LOGIN --- */
        .btn-login {
            width: 100%;
            padding: 14px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            font-family: var(--font-family);
            color: #000;
            background-color: var(--color-primary);
            cursor: pointer;
            transition: background-color 0.3s ease, transform 0.2s ease, box-shadow 0.3s ease;
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
        }

        .btn-login:hover:not(:disabled) {
            background-color: var(--color-primary-dark);
            transform: translateY(-2px);
            box-shadow: 0px 5px 15px rgba(120, 227, 32, 0.3);
        }

        .btn-login:disabled {
            background-color: #555;
            color: #999;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .loading-icon {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* --- PIE DE PÁGINA DENTRO DE LA TARJETA --- */
        .login-footer {
            margin-top: 30px; /* Espacio sobre el footer */
            text-align: center;
            font-size: 13px;
            color: var(--color-text-secondary);
        }

        /* --- RESPONSIVE DESIGN --- */
        @media (max-width: 480px) {
            body {
                padding: 15px;
            }
            .login-container {
                padding: 25px;
            }
            .login-header h2 {
                font-size: 20px;
            }
            .login-header p {
                font-size: 14px;
            }
            .login-header .logo-img {
                max-width: 120px;
            }
            .input-wrapper input, .btn-login {
                font-size: 15px;
                padding-top: 12px;
                padding-bottom: 12px;
            }
        }
    </style>
</head>
<body>

    <canvas id="background-canvas"></canvas>

    <div class="login-container">
        <div class="login-header">
            <img src="logo.png" alt="Logo AMI - Asistencia Médica Inmediata" class="logo-img">
            
            <h2>Panel de Control AMI</h2>
            <p>Acceso seguro para la gestión de Asistencia Médica Inmediata</p>
        </div>

        <form id="loginForm" novalidate>
            <div id="errorMessage" class="error-message"></div>
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token, ENT_QUOTES, 'UTF-8'); ?>">

            <div class="input-group">
                <label for="usuario">Usuario</label>
                <div class="input-wrapper">
                    <input type="text" id="usuario" name="usuario" required autocomplete="username" placeholder="Ingrese su nombre de usuario">
                    <i class="input-icon fa-solid fa-user"></i>
                </div>
            </div>

            <div class="input-group">
                <label for="password">Contraseña</label>
                <div class="input-wrapper">
                    <input type="password" id="password" name="password" required autocomplete="current-password" placeholder="Ingrese su contraseña">
                    <i class="input-icon fa-solid fa-lock"></i>
                </div>
            </div>

            <button type="submit" id="loginButton" class="btn-login">
                <span class="btn-text">Ingresar al Sistema</span>
                <i class="fa-solid fa-spinner loading-icon" style="display: none;"></i>
            </button>
        </form>

        <div class="login-footer">
            <p>&copy; <?php echo date('Y'); ?> AMI - Asistencia Médica Inmediata. Todos los derechos reservados.</p>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const canvas = document.getElementById('background-canvas');
        const ctx = canvas.getContext('2d');
        let particles = [];
        let mouse = { x: 0, y: 0 };
        const numParticles = 120;

        const medicalIcons = [
            '\uf7aa', '\uf484', '\uf21e', '\uf0fe', '\uf0f1', '\uf52f',
            '\uf48e', '\uf0f8', '\uf004', '\uf0c3', '\uf461', '\uf1fd'
        ];

        canvas.addEventListener('mousemove', function(e) {
            const rect = canvas.getBoundingClientRect();
            mouse.x = e.clientX - rect.left;
            mouse.y = e.clientY - rect.top;
        });

        function resizeCanvas() {
            canvas.width = window.innerWidth;
            canvas.height = window.innerHeight;
            particles = [];
            initParticles();
        }

        class MedicalParticle {
            constructor() {
                this.x = Math.random() * canvas.width;
                this.y = Math.random() * canvas.height;
                this.size = Math.random() * 16 + 12;
                this.speedX = (Math.random() - 0.5) * 0.5;
                this.speedY = (Math.random() - 0.5) * 0.5;
                this.icon = medicalIcons[Math.floor(Math.random() * medicalIcons.length)];
                this.opacity = Math.random() * 0.6 + 0.4;
                this.rotation = Math.random() * Math.PI * 2;
                this.rotationSpeed = (Math.random() - 0.5) * 0.01;
                this.pulse = Math.random() * Math.PI * 2;
                this.pulseSpeed = Math.random() * 0.02 + 0.01;
            }

            update() {
                const mouseDistance = Math.sqrt((mouse.x - this.x) ** 2 + (mouse.y - this.y) ** 2);
                if (mouseDistance < 100) {
                    const attractionForce = (100 - mouseDistance) / 100;
                    this.speedX += (mouse.x - this.x) * attractionForce * 0.0005;
                    this.speedY += (mouse.y - this.y) * attractionForce * 0.0005;
                }
                
                this.x += this.speedX;
                this.y += this.speedY;
                this.rotation += this.rotationSpeed;
                this.pulse += this.pulseSpeed;
                
                if (this.x <= 20 || this.x >= canvas.width - 20) { this.speedX *= -0.8; }
                if (this.y <= 20 || this.y >= canvas.height - 20) { this.speedY *= -0.8; }
                
                this.x = Math.max(20, Math.min(canvas.width - 20, this.x));
                this.y = Math.max(20, Math.min(canvas.height - 20, this.y));
                
                this.speedX *= 0.99;
                this.speedY *= 0.99;
            }

            draw() {
                ctx.save();
                ctx.translate(this.x, this.y);
                ctx.rotate(this.rotation);
                const pulseSize = this.size + Math.sin(this.pulse) * 2;
                ctx.font = `${pulseSize}px "Font Awesome 6 Free"`;
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                ctx.globalAlpha = this.opacity;
                ctx.fillStyle = '#78E320';
                ctx.shadowColor = 'rgba(120, 227, 32, 0.5)';
                ctx.shadowBlur = 8;
                ctx.fillText(this.icon, 0, 0);
                ctx.restore();
            }
        }

        function initParticles() {
            for (let i = 0; i < numParticles; i++) {
                particles.push(new MedicalParticle());
            }
        }

        function connectParticles() {
            for (let a = 0; a < particles.length; a++) {
                for (let b = a + 1; b < particles.length; b++) {
                    const distance = Math.sqrt((particles[a].x - particles[b].x) ** 2 + (particles[a].y - particles[b].y) ** 2);
                    let opacity = 0;
                    if (distance < 150) opacity = (1 - distance / 150) * 0.4;
                    else if (distance < 300) opacity = (1 - distance / 300) * 0.2;
                    else opacity = 0.05;
                    
                    if (opacity > 0) {
                        ctx.strokeStyle = `rgba(120, 227, 32, ${opacity})`;
                        ctx.lineWidth = opacity > 0.2 ? 1.5 : 0.5;
                        ctx.beginPath();
                        ctx.moveTo(particles[a].x, particles[a].y);
                        ctx.lineTo(particles[b].x, particles[b].y);
                        ctx.stroke();
                    }
                }
                
                const mouseDistance = Math.sqrt((mouse.x - particles[a].x) ** 2 + (mouse.y - particles[a].y) ** 2);
                if (mouseDistance < 120) {
                    const opacity = (1 - mouseDistance / 120) * 0.6;
                    ctx.strokeStyle = `rgba(120, 227, 32, ${opacity})`;
                    ctx.lineWidth = 2;
                    ctx.beginPath();
                    ctx.moveTo(particles[a].x, particles[a].y);
                    ctx.lineTo(mouse.x, mouse.y);
                    ctx.stroke();
                }
            }
        }

        function animateParticles() {
            requestAnimationFrame(animateParticles);
            
            const gradient = ctx.createLinearGradient(0, 0, canvas.width, canvas.height);
            gradient.addColorStop(0, '#1C1C1C');
            gradient.addColorStop(1, '#000000');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            connectParticles();
            
            particles.forEach(p => { p.update(); p.draw(); });
        }

        window.addEventListener('resize', resizeCanvas);
        resizeCanvas();
        animateParticles();

        // --- Lógica de SweetAlert2 y Formulario (simplificada para brevedad) ---
        const loginForm = document.getElementById('loginForm');
        const errorMessage = document.getElementById('errorMessage');
        const loginButton = document.getElementById('loginButton');
        const btnText = loginButton.querySelector('.btn-text');
        const loadingIcon = loginButton.querySelector('.loading-icon');

        async function attemptLogin(force = false) {
            errorMessage.textContent = '';
            btnText.textContent = 'Verificando...';
            loadingIcon.style.display = 'inline-block';
            loginButton.disabled = true;

            const formData = new FormData(loginForm);
            if (force) formData.append('force_login', 'true');

            try {
                const response = await fetch('/api/validar_login.php', { method: 'POST', body: formData });
                const data = await response.json();

                if (data.success) {
                    window.location.href = data.redirectUrl;
                } else if (data.error_code === 'SESSION_ALREADY_ACTIVE') {
                    Swal.fire({
                        title: 'Sesión Activa Detectada',
                        html: `Ya tienes una sesión abierta.<br><small>Última actividad desde la IP: ${data.session_details.ip}</small>`,
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonText: 'Sí, cerrar y continuar aquí',
                        cancelButtonText: 'Cancelar',
                        confirmButtonColor: '#78E320',
                        cancelButtonColor: '#6c757d'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            attemptLogin(true);
                        } else {
                            resetButton();
                        }
                    });
                } else {
                    errorMessage.textContent = data.message || 'Credenciales incorrectas.';
                    resetButton();
                }
            } catch (error) {
                errorMessage.textContent = 'Error de conexión con el servidor.';
                resetButton();
            }
        }

        function resetButton() {
            btnText.textContent = 'Ingresar al Sistema';
            loadingIcon.style.display = 'none';
            loginButton.disabled = false;
        }

        loginForm.addEventListener('submit', (event) => {
            event.preventDefault();
            attemptLogin(false);
        });
    });
    </script>
</body>
</html>