<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php'; // seguridad.php ya nos da el nombre del usuario en $_SESSION['user_nombre']

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;
// Obtenemos el nombre del operador para usarlo en los logs detallados
$operador_nombre = $_SESSION['user_nombre'] ?? 'Sistema';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']))) {
    $response['message'] = 'Error de seguridad (CSRF).';
    echo json_encode($response);
    exit();
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'buscar_clientes':
        $term = $_GET['q'] ?? '';
        $stmt = $conn->prepare("
            SELECT 
                id, 
                CONCAT(nombre, ' (C.I: ', IFNULL(cedula, 'N/A'), ')') as text,
                CASE WHEN cedula IS NULL OR cedula = '' THEN 1 ELSE 0 END as needs_cedula
            FROM clientes 
            WHERE nombre LIKE ? OR cedula LIKE ? 
            LIMIT 10
        ");
        $stmt->execute(["%{$term}%", "%{$term}%"]);
        $response = ['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
        break;

    case 'buscar_productos':
        $term = $_GET['q'] ?? '';
        $stmt = $conn->prepare("
            SELECT p.id, CONCAT(p.nombre, ' (Stock: ', s.cantidad, ')') as text 
            FROM almacen_productos p 
            JOIN almacen_stock s ON p.id = s.producto_id 
            WHERE (p.nombre LIKE ? OR p.sku LIKE ?) 
            AND s.cantidad > 0 
            LIMIT 10
        ");
        $stmt->execute(["%{$term}%", "%{$term}%"]);
        $response = ['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)];
        break;

    case 'crear_cliente':
        try {
            $nombre = trim($_POST['nombre'] ?? '');
            $cedula = trim($_POST['cedula'] ?? '');
            $telefono = trim($_POST['telefono'] ?? '');

            if (empty($nombre)) throw new Exception("El nombre del cliente es obligatorio.");
            if (empty($cedula)) throw new Exception("La cédula del cliente es obligatoria.");
            if (empty($telefono)) throw new Exception("El teléfono del cliente es obligatorio.");

            $stmt_check = $conn->prepare("SELECT id FROM clientes WHERE cedula = ? OR telefono = ? LIMIT 1");
            $stmt_check->execute([$cedula, $telefono]);
            if ($stmt_check->fetch()) {
                throw new Exception("Ya existe un cliente registrado con la misma cédula o teléfono.");
            }
            
            $stmt = $conn->prepare("INSERT INTO clientes (nombre, cedula, telefono, creado_por_id) VALUES (?, ?, ?, ?)");
            $stmt->execute([$nombre, $cedula, $telefono, $usuario_id]);
            $cliente_id = $conn->lastInsertId();
            
            // --- LOG DETALLADO ---
            $detalle_log = "El operador '{$operador_nombre}' registró al nuevo cliente '{$nombre}' con C.I. {$cedula}.";
            registrar_accion('crear_cliente', $detalle_log);

            $response = [
                'success' => true,
                'message' => 'Cliente creado con éxito',
                'cliente' => ['id' => $cliente_id, 'text' => $nombre . ' (C.I: ' . $cedula . ')']
            ];
        } catch (Exception $e) {
            $response['message'] = $e->getMessage();
        }
        break;

    case 'actualizar_cliente_cedula':
        try {
            $cliente_id = (int) ($_POST['cliente_id'] ?? 0);
            $cedula = trim($_POST['cedula'] ?? '');

            if (empty($cedula) || $cliente_id === 0) throw new Exception("Datos incompletos.");
            
            $stmt_check = $conn->prepare("SELECT id FROM clientes WHERE cedula = ? AND id != ? LIMIT 1");
            $stmt_check->execute([$cedula, $cliente_id]);
            if ($stmt_check->fetch()) {
                throw new Exception("La cédula ingresada ya pertenece a otro cliente.");
            }

            $stmt_nombre = $conn->prepare("SELECT nombre FROM clientes WHERE id = ?");
            $stmt_nombre->execute([$cliente_id]);
            $nombre_cliente = $stmt_nombre->fetchColumn();

            $stmt = $conn->prepare("UPDATE clientes SET cedula = ? WHERE id = ?");
            $stmt->execute([$cedula, $cliente_id]);
            
            // --- LOG DETALLADO ---
            $detalle_log = "El operador '{$operador_nombre}' actualizó la cédula del cliente '{$nombre_cliente}' (ID: {$cliente_id}) a la nueva cédula {$cedula}.";
            registrar_accion('actualizar_cedula_cliente', $detalle_log);

            $response = [
                'success' => true, 
                'message' => 'Cédula actualizada',
                'new_text' => $nombre_cliente . ' (C.I: ' . $cedula . ')'
            ];
        } catch (Exception $e) {
            $response['message'] = $e->getMessage();
        }
        break;

    case 'registrar_venta':
        try {
            $conn->beginTransaction();

            // Lógica de subida de comprobante (sin cambios)
            $comprobante_path = null;
            if (isset($_FILES['comprobante_pago']) && $_FILES['comprobante_pago']['error'] == 0) {
                 $upload_dir = '../uploads/comprobantes/';
                 if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
                 $ext = pathinfo($_FILES['comprobante_pago']['name'], PATHINFO_EXTENSION);
                 $comprobante_path = 'comp_' . uniqid() . '.' . $ext;
                 if (!move_uploaded_file($_FILES['comprobante_pago']['tmp_name'], $upload_dir . $comprobante_path)) {
                     throw new Exception('Error al guardar el comprobante.');
                 }
            }

            // Inserción de la venta (sin cambios)
            $sql_venta = "INSERT INTO ventas (operador_id, cliente_id, fecha_venta, monto_total, metodo_pago, referencia_pago, comprobante_pago, direccion_entrega, ciudad, estado_ve, observaciones, estado) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pendiente')";
            $stmt_venta = $conn->prepare($sql_venta);
            $stmt_venta->execute([$usuario_id, $_POST['cliente_id'], $_POST['fecha_venta'], $_POST['monto_total'], $_POST['metodo_pago'], $_POST['referencia_pago'], $comprobante_path, $_POST['direccion_entrega'], $_POST['ciudad'], $_POST['estado_ve'], $_POST['observaciones']]);
            $venta_id = $conn->lastInsertId();

            $productos = $_POST['productos'] ?? [];
            if (empty($productos)) {
                throw new Exception("No se han añadido productos a la venta.");
            }
            
            // --- LÓGICA PARA EL LOG DETALLADO DE VENTA ---
            $ids_productos = array_column($productos, 'id');
            $placeholders = implode(',', array_fill(0, count($ids_productos), '?'));
            $stmt_nombres = $conn->prepare("SELECT id, nombre FROM almacen_productos WHERE id IN ($placeholders)");
            $stmt_nombres->execute($ids_productos);
            $nombres_map = $stmt_nombres->fetchAll(PDO::FETCH_KEY_PAIR);
            $productos_log_array = [];
            // --- FIN LÓGICA LOG ---

            $sql_item = "INSERT INTO venta_items (venta_id, producto_id, cantidad, precio_unitario, subtotal) VALUES (?, ?, ?, ?, ?)";
            $stmt_item = $conn->prepare($sql_item);
            
            $sql_update_stock = "UPDATE almacen_stock SET cantidad = cantidad - ? WHERE producto_id = ?";
            $stmt_update_stock = $conn->prepare($sql_update_stock);

            $sql_movimiento = "INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, referencia_id, usuario_id) VALUES (?, 'egreso', ?, ?, ?, ?)";
            $stmt_movimiento = $conn->prepare($sql_movimiento);

            foreach ($productos as $p) {
                $producto_id = $p['id'];
                $cantidad = (int) $p['cantidad'];
                $precio = (float) $p['precio'];
                $subtotal = $cantidad * $precio;
                
                $stmt_item->execute([$venta_id, $producto_id, $cantidad, $precio, $subtotal]);
                $stmt_update_stock->execute([$cantidad, $producto_id]);
                // El movimiento de almacén puede ser menos descriptivo que el log principal
                $stmt_movimiento->execute([$producto_id, $cantidad, "Venta #{$venta_id}", $venta_id, $usuario_id]);
                
                // Construir la lista de productos para el log
                $nombre_producto = $nombres_map[$producto_id] ?? 'ID ' . $producto_id;
                $productos_log_array[] = "{$nombre_producto} (x{$cantidad})";
            }
            
            $conn->commit();

            // --- REGISTRO FINAL EN BITÁCORA ---
            $productos_string = implode(', ', $productos_log_array);
            $detalle_log = "El operador '{$operador_nombre}' registró la venta #{$venta_id} con los siguientes productos: {$productos_string}.";
            registrar_accion('registro_venta', $detalle_log);
            
            $response = ['success' => true, 'message' => "Venta #{$venta_id} registrada con éxito. Pasará a despacho para su verificación."];

        } catch (Exception $e) {
            if ($conn->inTransaction()) {
                $conn->rollBack();
            }
            $response['message'] = 'Error al registrar la venta: ' . $e->getMessage();
            error_log('Error en registrar_venta: ' . $e->getMessage());
        }
        break;
}

echo json_encode($response);