<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';
require_once 'lib/fpdf.php';

// --- CLASE PDF PERSONALIZADA CON LOS NUEVOS AJUSTES ---
class PDF_Reporte_Comisiones extends FPDF {
    private $Periodo;

    function SetPeriodo($periodo) {
        $this->Periodo = $periodo;
    }

    function Header() {
        if (file_exists('../assets/images/logo.png')) {
            // Ajustamos el logo para el formato horizontal
            $this->Image('../assets/images/logo.png', 10, 8, 40);
        }
        $this->SetFont('Arial', 'B', 16);
        $this->Cell(0, 10, utf8_decode('Reporte de Comisiones por Operador'), 0, 1, 'C');
        $this->SetFont('Arial', '', 11);
        $this->Cell(0, 7, utf8_decode($this->Periodo), 0, 1, 'C');
        $this->Ln(8);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetFont('Arial', 'I', 8);
        $this->Cell(0, 10, utf8_decode('Página ') . $this->PageNo() . '/{nb}', 0, 0, 'C');
    }

    function TablaHeaderOperador() {
        $this->SetFillColor(27, 94, 32); // Verde Oscuro
        $this->SetTextColor(255);
        $this->SetDrawColor(76, 175, 80);
        $this->SetFont('Arial', 'B', 9);
        // Columnas ajustadas para el ancho de página Legal (330mm aprox de área útil con márgenes)
        $this->Cell(70, 7, 'Ruta', 1, 0, 'C', true);
        $this->Cell(70, 7, 'Jornada', 1, 0, 'C', true);
        $this->Cell(30, 7, 'Pac. Atendidos', 1, 0, 'C', true);
        $this->Cell(40, 7, utf8_decode('Comisión ($)'), 1, 0, 'C', true);
        $this->Cell(40, 7, 'Venta Med. ($)', 1, 0, 'C', true);
        $this->Cell(40, 7, utf8_decode('Comisión Med. ($)'), 1, 0, 'C', true);
        $this->Cell(40, 7, 'Total Neto ($)', 1, 1, 'C', true);
    }
}

try {
    // 1. OBTENER PARÁMETROS Y VALIDAR
    $fecha_inicio = $_GET['fecha_inicio'] ?? '';
    $fecha_fin = $_GET['fecha_fin'] ?? '';
    $tasa_bsd = (float)($_GET['tasa_bsd'] ?? 0);

    if (!preg_match("/^\d{4}-\d{2}-\d{2}$/", $fecha_inicio) || !preg_match("/^\d{4}-\d{2}-\d{2}$/", $fecha_fin) || $tasa_bsd <= 0) {
        throw new Exception("Parámetros inválidos (fechas o tasa de cambio).");
    }

    // 2. CARGAR REGLAS DE NEGOCIO
    $stmt_config = $conn->query("SELECT clave, valor FROM config_comisiones");
    $config_raw = $stmt_config->fetchAll(PDO::FETCH_KEY_PAIR);
    $config = [
        'paciente_umbral' => (int)($config_raw['paciente_umbral_cantidad'] ?? 18),
        'paciente_comision' => (float)($config_raw['paciente_comision_monto'] ?? 2.00),
        'medicamento_umbral' => (float)($config_raw['medicamento_umbral_venta'] ?? 500.00),
        'medicamento_porcentaje' => (float)($config_raw['medicamento_comision_porcentaje'] ?? 5.00)
    ];

    // 3. LA CONSULTA MAESTRA (SIN CAMBIOS)
    $sql = "
        SELECT r.nombre_ruta, j.id AS jornada_id, j.ciudad_nombre, j.fecha_jornada, u.id AS operador_id, u.nombre AS operador_nombre,
        COUNT(jc.id) AS pacientes_atendidos_por_operador,
        COALESCE(SUM(jm.cantidad * jm.precio), 0) AS venta_medicamentos_por_operador,
        (SELECT COUNT(*) FROM jornada_clientes WHERE jornada_id = j.id AND estado_cita = 'Atendido') AS total_pacientes_jornada,
        (SELECT COALESCE(SUM(jmed.cantidad * jmed.precio), 0) FROM jornada_medicamentos jmed JOIN jornada_clientes jcli ON jmed.jornada_cliente_id = jcli.id WHERE jcli.jornada_id = j.id) AS total_venta_medicamentos_jornada
        FROM jornadas j JOIN rutas r ON j.ruta_id = r.id JOIN jornada_clientes jc ON j.id = jc.jornada_id
        LEFT JOIN jornada_medicamentos jm ON jc.id = jm.jornada_cliente_id
        JOIN usuarios u ON jc.agendado_por_operador_id = u.id
        WHERE j.fecha_jornada BETWEEN :fecha_inicio AND :fecha_fin AND r.archivada = 0 AND jc.estado_cita = 'Atendido'
        GROUP BY r.nombre_ruta, j.id, j.ciudad_nombre, j.fecha_jornada, u.id, u.nombre
        ORDER BY j.fecha_jornada, r.nombre_ruta, u.nombre;
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute(['fecha_inicio' => $fecha_inicio, 'fecha_fin' => $fecha_fin]);
    $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($resultados)) {
        throw new Exception("No se encontraron datos de jornadas atendidas en el período seleccionado.");
    }
    
    // 4. PROCESAR DATOS
    $reporte_data = [];
    $jornadas_summary = [];
    foreach ($resultados as $row) {
        $operador_id = $row['operador_id'];
        $jornada_id = $row['jornada_id'];
        if (!isset($jornadas_summary[$jornada_id])) {
            $jornadas_summary[$jornada_id] = ['nombre' => $row['ciudad_nombre'] . ' (' . date('d/m/Y', strtotime($row['fecha_jornada'])) . ')','ruta' => $row['nombre_ruta'],'total_pacientes' => (int)$row['total_pacientes_jornada'],'total_venta_medicamentos' => (float)$row['total_venta_medicamentos_jornada']];
        }
        if (!isset($reporte_data[$operador_id])) {
            $reporte_data[$operador_id] = ['nombre' => $row['operador_nombre'], 'jornadas' => [], 'total_comision_pacientes' => 0, 'total_comision_medicamentos' => 0, 'gran_total' => 0];
        }
        $comision_pacientes = ($row['total_pacientes_jornada'] >= $config['paciente_umbral']) ? $row['pacientes_atendidos_por_operador'] * $config['paciente_comision'] : 0;
        $comision_medicamentos = ($row['total_venta_medicamentos_jornada'] >= $config['medicamento_umbral']) ? $row['venta_medicamentos_por_operador'] * ($config['medicamento_porcentaje'] / 100) : 0;
        $total_neto_jornada = $comision_pacientes + $comision_medicamentos;
        $reporte_data[$operador_id]['jornadas'][] = ['nombre_ruta' => $row['nombre_ruta'], 'ciudad_nombre' => $row['ciudad_nombre'] . ' (' . date('d/m/Y', strtotime($row['fecha_jornada'])) . ')', 'pacientes_atendidos' => $row['pacientes_atendidos_por_operador'], 'comision_pacientes' => $comision_pacientes, 'venta_medicamentos' => $row['venta_medicamentos_por_operador'], 'comision_medicamentos' => $comision_medicamentos, 'total_neto' => $total_neto_jornada];
        $reporte_data[$operador_id]['total_comision_pacientes'] += $comision_pacientes;
        $reporte_data[$operador_id]['total_comision_medicamentos'] += $comision_medicamentos;
        $reporte_data[$operador_id]['gran_total'] += $total_neto_jornada;
    }

    // 5. GENERAR EL PDF
    $periodo_str = "Periodo del " . date('d/m/Y', strtotime($fecha_inicio)) . " al " . date('d/m/Y', strtotime($fecha_fin)) . " (Tasa: " . number_format($tasa_bsd, 2, ',', '.') . " Bs./USD)";
    $pdf = new PDF_Reporte_Comisiones('L', 'mm', 'Legal'); // Oficio Horizontal
    $pdf->SetMargins(10, 10, 10); // Márgenes estrechos
    $pdf->SetPeriodo($periodo_str);
    $pdf->AliasNbPages();
    $pdf->AddPage();

    // Resumen General de Jornadas
    $pdf->SetFont('Arial', 'B', 12);
    $pdf->Cell(0, 10, utf8_decode('Resumen General de Jornadas en el Período'), 0, 1, 'L');
    $pdf->SetFont('Arial', 'B', 10);
    $pdf->SetFillColor(46, 125, 50); // Verde oscuro
    $pdf->SetTextColor(255);
    $pdf->Cell(140, 7, 'Jornada', 1, 0, 'C', true);
    $pdf->Cell(65, 7, 'Total Pacientes Atendidos', 1, 0, 'C', true);
    $pdf->Cell(65, 7, 'Total Venta Medicamentos ($)', 1, 1, 'C', true);
    $pdf->SetFont('Arial', '', 10);
    $pdf->SetTextColor(0);
    foreach($jornadas_summary as $j) {
        $pdf->Cell(140, 7, utf8_decode($j['ruta'] . ' - ' . $j['nombre']), 1);
        $pdf->Cell(65, 7, $j['total_pacientes'], 1, 0, 'C');
        $pdf->Cell(65, 7, '$' . number_format($j['total_venta_medicamentos'], 2, ',', '.'), 1, 1, 'R');
    }
    $pdf->Ln(8);

    // Desglose por Operador
    $pdf->SetFont('Arial', 'B', 12);
    $pdf->Cell(0, 10, utf8_decode('Desglose de Comisiones por Operador'), 0, 1, 'L');
    
    $gran_total_general_usd = 0;

    foreach ($reporte_data as $operador) {
        if ($pdf->GetY() > 155) { // Check de espacio para evitar cortes feos
            $pdf->AddPage();
        }
        $pdf->SetFont('Arial', 'B', 11);
        $pdf->SetFillColor(76, 175, 80); // Verde medio
        $pdf->SetTextColor(255);
        $pdf->Cell(0, 8, utf8_decode('Operador: ' . $operador['nombre']), 1, 1, 'L', true);
        $pdf->TablaHeaderOperador();
        
        $pdf->SetFont('Arial', '', 9);
        $pdf->SetTextColor(0);

        foreach($operador['jornadas'] as $jornada) {
            $pdf->Cell(70, 6, utf8_decode($jornada['nombre_ruta']), 1);
            $pdf->Cell(70, 6, utf8_decode($jornada['ciudad_nombre']), 1);
            $pdf->Cell(30, 6, $jornada['pacientes_atendidos'], 1, 0, 'C');
            $pdf->Cell(40, 6, number_format($jornada['comision_pacientes'], 2, ',', '.'), 1, 0, 'R');
            $pdf->Cell(40, 6, number_format($jornada['venta_medicamentos'], 2, ',', '.'), 1, 0, 'R');
            $pdf->Cell(40, 6, number_format($jornada['comision_medicamentos'], 2, ',', '.'), 1, 0, 'R');
            $pdf->Cell(40, 6, number_format($jornada['total_neto'], 2, ',', '.'), 1, 1, 'R');
        }

        // Totales por operador
        $total_bs = $operador['gran_total'] * $tasa_bsd;
        $gran_total_general_usd += $operador['gran_total'];
        $pdf->SetFont('Arial', 'B', 10);
        $pdf->SetFillColor(220, 240, 220); // Verde muy claro
        $pdf->Cell(210, 8, 'TOTALES A PAGAR AL OPERADOR', 1, 0, 'R', true);
        $pdf->Cell(120, 8, utf8_decode('$' . number_format($operador['gran_total'], 2, ',', '.') . '  /  Bs. ' . number_format($total_bs, 2, ',', '.')), 1, 1, 'C', true);
        $pdf->Ln(6);
    }
    
    // GRAN TOTAL GENERAL
    $pdf->Ln(5);
    $gran_total_general_bs = $gran_total_general_usd * $tasa_bsd;
    $pdf->SetFont('Arial', 'B', 12);
    $pdf->SetFillColor(27, 94, 32); // Verde Oscuro
    $pdf->SetTextColor(255);
    $pdf->Cell(210, 10, 'GRAN TOTAL GENERAL DE COMISIONES (TODOS LOS OPERADORES)', 1, 0, 'R', true);
    $pdf->Cell(120, 10, utf8_decode('$' . number_format($gran_total_general_usd, 2, ',', '.') . '  /  Bs. ' . number_format($gran_total_general_bs, 2, ',', '.')), 1, 1, 'C', true);
    
    $pdf->Output('D', 'Reporte_Comisiones_' . date('Y-m-d') . '.pdf');

} catch (Exception $e) {
    header('Content-Type: application/json', true, 500);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>