<?php
// Mostrar errores para depuración
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../administracion/includes/seguridad.php';
require_once 'lib/fpdf.php';

class PDF_Despacho extends FPDF {
    
    // Función para convertir texto con tildes
    function TextoUTF8($texto) {
        return iconv('UTF-8', 'ISO-8859-1//IGNORE', $texto);
    }
    
    // Constructor para tamaño oficio
    function __construct() {
        parent::__construct('P', 'mm', 'Legal'); // Tamaño Legal/Oficio (216 x 356 mm)
    }
    
    // Header simplificado para formato compacto
    function HeaderCompacto($y_pos, $tipo = 'ORIGINAL') {
        // Fondo del header
        $this->SetFillColor(240, 255, 240);
        $this->Rect(0, $y_pos, 216, 25, 'F');
        
        // Borde superior
        $this->SetFillColor(27, 94, 32);
        $this->Rect(0, $y_pos, 216, 2, 'F');
        
        // Logo (más pequeño)
        if (file_exists('../assets/images/logo.png')) {
            $this->Image('../assets/images/logo.png', 10, $y_pos + 3, 20);
        }
        
        // Título principal
        $this->SetFont('Arial', 'B', 14);
        $this->SetTextColor(27, 94, 32);
        $this->SetXY(40, $y_pos + 5);
        $this->Cell(120, 6, $this->TextoUTF8('ORDEN DE DESPACHO - ' . $tipo), 0, 0, 'C');
        
        // Línea separadora
        $this->SetXY(40, $y_pos + 13);
        $this->SetFont('Arial', '', 8);
        $this->SetTextColor(76, 175, 80);
        $this->Cell(120, 4, $this->TextoUTF8('Control de Envíos y Entregas'), 0, 0, 'C');
        
        // Tipo de copia en la esquina
        $this->SetFont('Arial', 'B', 10);
        $this->SetTextColor(255, 255, 255);
        $this->SetFillColor(139, 195, 74);
        $this->SetXY(170, $y_pos + 3);
        $this->Cell(40, 8, $tipo, 0, 0, 'C', true);
        
        // Línea decorativa
        $this->SetDrawColor(139, 195, 74);
        $this->SetLineWidth(1);
        $this->Line(10, $y_pos + 20, 206, $y_pos + 20);
        
        // Resetear valores
        $this->SetTextColor(0, 0, 0);
        $this->SetDrawColor(0, 0, 0);
        $this->SetLineWidth(0.2);
    }
    
    // Método para títulos de sección compactos
    function SeccionTituloCompacto($titulo, $y_pos) {
        $this->SetFillColor(76, 175, 80);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('Arial', 'B', 9);
        $this->SetXY(10, $y_pos);
        $this->Cell(196, 6, $this->TextoUTF8($titulo), 0, 1, 'L', true); // Ancho ajustado a 196
        $this->SetTextColor(0, 0, 0);
    }
    
    // Método para campos compactos
    function CampoCompacto($x, $y, $etiqueta, $valor, $ancho = 90) {
        $this->SetXY($x, $y);
        $this->SetFont('Arial', 'B', 8);
        $this->SetTextColor(27, 94, 32);
        $etiqueta_ancho = $this->GetStringWidth($etiqueta) + 2;
        $this->Cell($etiqueta_ancho, 4, $this->TextoUTF8($etiqueta));
        
        $this->SetFont('Arial', '', 8);
        $this->SetTextColor(0, 0, 0);
        $this->Cell($ancho - $etiqueta_ancho, 4, $this->TextoUTF8($valor));
    }
    
    // Generar una orden de despacho compacta
    function GenerarOrdenCompacta($venta, $items, $y_inicio, $tipo = 'ORIGINAL') {
        // Header
        $this->HeaderCompacto($y_inicio, $tipo);
        $y = $y_inicio + 25;
        
        // Información General
        $this->SeccionTituloCompacto('INFORMACIÓN GENERAL', $y);
        $y += 8;
        
        $this->SetFillColor(245, 255, 245);
        $this->Rect(10, $y, 196, 12, 'F');
        
        $this->CampoCompacto(15, $y + 2, 'Venta: ', '#' . str_pad($venta['id'], 6, '0', STR_PAD_LEFT), 70);
        $this->CampoCompacto(90, $y + 2, 'Fecha: ', date('d/m/Y', strtotime($venta['fecha_venta'])), 60);
        $this->CampoCompacto(155, $y + 2, 'Hora: ', date('H:i'), 50);
        
        $this->CampoCompacto(15, $y + 7, 'Asesor: ', $venta['operador_nombre'], 180);
        
        $y += 15;
        
        // Datos de Envío
        $this->SeccionTituloCompacto('DATOS PARA EL ENVÍO', $y);
        $y += 8;
        
        $this->SetFillColor(240, 255, 240);
        $this->SetDrawColor(76, 175, 80);
        $this->Rect(10, $y, 196, 20, 'FD');
        
        $this->CampoCompacto(15, $y + 2, 'Cliente: ', $venta['cliente_nombre'], 100);
        $this->CampoCompacto(120, $y + 2, 'Cédula: ', $venta['cedula'], 80);
        
        $this->CampoCompacto(15, $y + 7, 'Teléfono: ', $venta['telefono'], 100);
        
        // Dirección en una sola línea
        $direccion_completa = $venta['direccion_entrega'] . ', ' . $venta['ciudad'] . ', ' . $venta['estado_ve'];
        $this->SetXY(15, $y + 12);
        $this->SetFont('Arial', 'B', 8);
        $this->SetTextColor(27, 94, 32);
        $this->Cell(20, 4, $this->TextoUTF8('Dirección: '));
        $this->SetFont('Arial', '', 8);
        $this->SetTextColor(0, 0, 0);
        $this->Cell(0, 4, $this->TextoUTF8(substr($direccion_completa, 0, 85)));
        
        $y += 25;
        
        // Productos
        $this->SeccionTituloCompacto('PRODUCTOS A DESPACHAR', $y);
        $y += 8;
        
        // =========================================================================
        // ======================= INICIO DE MODIFICACIONES ========================
        // =========================================================================

        // Header de tabla con precios
        $this->SetFillColor(76, 175, 80);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('Arial', 'B', 8);
        
        $this->SetXY(10, $y);
        $this->Cell(100, 6, $this->TextoUTF8('DESCRIPCIÓN DEL PRODUCTO'), 1, 0, 'C', true);
        $this->Cell(20, 6, 'CANT.', 1, 0, 'C', true);
        $this->Cell(38, 6, 'P. UNITARIO', 1, 0, 'C', true);
        $this->Cell(38, 6, 'SUBTOTAL', 1, 1, 'C', true);
        
        $y += 6;
        
        // Productos con precios
        $this->SetTextColor(0, 0, 0);
        $this->SetFont('Arial', '', 8);
        
        $max_productos = 5; // Máximo 5 productos para que quepa el total
        $productos_mostrados = 0;
        
        foreach($items as $index => $item) {
            if ($productos_mostrados >= $max_productos) break;
            
            $fill = ($index % 2 == 0) ? $this->SetFillColor(250, 255, 250) : $this->SetFillColor(245, 255, 245);
            
            $this->SetXY(10, $y);
            $this->Cell(100, 5, $this->TextoUTF8(substr($item['producto_nombre'], 0, 45)), 1, 0, 'L', true);
            $this->Cell(20, 5, $item['cantidad'], 1, 0, 'C', true);
            $this->Cell(38, 5, '$ ' . number_format($item['precio_unitario'], 2, ',', '.'), 1, 0, 'R', true);
            $this->Cell(38, 5, '$ ' . number_format($item['subtotal'], 2, ',', '.'), 1, 1, 'R', true);
            
            $productos_mostrados++;
            $y += 5;
        }
        
        // Si hay más productos, indicarlo
        if (count($items) > $max_productos) {
            $this->SetFillColor(255, 255, 200);
            $this->SetXY(10, $y);
            $this->Cell(196, 5, $this->TextoUTF8('... y ' . (count($items) - $max_productos) . ' productos más (ver sistema)'), 1, 1, 'C', true);
            $y += 5;
        }
        
        // Total General
        $this->SetFillColor(230, 245, 230);
        $this->SetFont('Arial', 'B', 9);
        $this->SetXY(10, $y);
        $this->Cell(158, 6, $this->TextoUTF8('TOTAL GENERAL:'), 1, 0, 'R', true);
        $this->SetFillColor(76, 175, 80);
        $this->SetTextColor(255, 255, 255);
        $this->Cell(38, 6, '$ ' . number_format($venta['monto_total'], 2, ',', '.'), 1, 1, 'R', true);
        
        $y += 9;
        
        // =======================================================================
        // ========================= FIN DE MODIFICACIONES =======================
        // =======================================================================
        
        // Observaciones compactas
        $this->SeccionTituloCompacto('OBSERVACIONES', $y);
        $y += 8;
        
        $this->SetFillColor(255, 248, 220);
        $this->Rect(10, $y, 196, 10, 'F');
        
        $observaciones = $venta['observaciones'] ?: 'Sin observaciones adicionales.';
        $this->SetXY(15, $y + 2);
        $this->SetFont('Arial', '', 8);
        $this->SetTextColor(120, 80, 0);
        $this->MultiCell(186, 3, $this->TextoUTF8(substr($observaciones, 0, 150)));
        
        // Línea separadora entre las dos copias (solo para la primera)
        if ($tipo == 'ORIGINAL') {
            $this->SetDrawColor(200, 200, 200);
            $this->SetLineWidth(0.5);
            $y_linea = $y_inicio + 170;
            for($x = 10; $x < 200; $x += 5) {
                $this->Line($x, $y_linea, $x + 2, $y_linea);
            }
            $this->SetDrawColor(0, 0, 0);
            $this->SetLineWidth(0.2);
        }
    }
    
    // Override del Footer para que no se superponga
    function Footer() {
        // No footer automático, lo manejamos manualmente
    }
}

if (!puede('gestionar_despachos')) {
    die('Acceso denegado.');
}

$venta_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$venta_id) {
    die('ID de venta no válido.');
}

try {
    // Consulta para obtener datos de la venta
    $stmt = $conn->prepare("
        SELECT v.*, c.nombre as cliente_nombre, c.cedula, c.telefono, u.nombre as operador_nombre
        FROM ventas v
        JOIN clientes c ON v.cliente_id = c.id
        JOIN usuarios u ON v.operador_id = u.id
        WHERE v.id = ?
    ");
    $stmt->execute([$venta_id]);
    $venta = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$venta) { 
        die('Venta no encontrada.'); 
    }
    
    // Obtener items de la venta
    $stmt_items = $conn->prepare("
        SELECT vi.cantidad, vi.precio_unitario, vi.subtotal, p.nombre as producto_nombre, p.sku
        FROM venta_items vi
        JOIN almacen_productos p ON vi.producto_id = p.id
        WHERE vi.venta_id = ?
    ");
    $stmt_items->execute([$venta_id]);
    $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);
    
    // Iniciar PDF
    $pdf = new PDF_Despacho();
    $pdf->AddPage();
    
    // Generar ORIGINAL (parte superior)
    $pdf->GenerarOrdenCompacta($venta, $items, 5, 'ORIGINAL');
    
    // Generar COPIA (parte inferior)
    $pdf->GenerarOrdenCompacta($venta, $items, 180, 'COPIA');
    
    // Footer al final de la página
    $pdf->SetFont('Arial', 'I', 7);
    $pdf->SetTextColor(100, 100, 100);
    $pdf->SetXY(10, 350);
    
    $pdf->Output('D', 'orden_despacho_doble_' . str_pad($venta_id, 6, '0', STR_PAD_LEFT) . '_' . date('Ymd_His') . '.pdf');
    
} catch (Exception $e) {
    // Mostrar error detallado
    echo "Error al generar el PDF:<br>";
    echo "Mensaje: " . $e->getMessage() . "<br>";
    echo "Archivo: " . $e->getFile() . "<br>";
    echo "Línea: " . $e->getLine() . "<br>";
    echo "Trace: <pre>" . $e->getTraceAsString() . "</pre>";
    die();
}
?>