<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida o sin permisos.'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

// === FUNCIÓN PARA BITÁCORA DETALLADA (ADAPTADA DE EMPLEADOS_AJAX) ===
function registrarBitacoraDetallada($conn, $accion, $detalle_especifico) {
    if (!isset($_SESSION['user_id'])) return;
    $usuario_admin = $_SESSION['user_nombre'] ?? 'Usuario Desconocido';
    $detalle_completo = "ADMIN: {$usuario_admin} - {$detalle_especifico}";
    registrar_accion($accion, $detalle_completo);
}

// === FUNCIÓN AUXILIAR PARA OBTENER NOMBRES ===
function obtenerNombreNomina($conn, $nomina_id) {
    $stmt = $conn->prepare("SELECT nombre_nomina FROM nominas WHERE id = ?");
    $stmt->execute([$nomina_id]);
    return $stmt->fetchColumn() ?: "ID {$nomina_id}";
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'cargar_datos_iniciales':
        if (!puede('gestionar_nomina')) break;
        registrarBitacoraDetallada($conn, 'acceso_modulo_nomina', 'Accedió al módulo de creación de Nóminas.');
        try {
            $stmt = $conn->prepare("
                SELECT 
                    u.id, u.nombre, u.cedula, u.salario_base, r.nombre as cargo,
                    IFNULL(nc.cesta_ticket, 40.00) as cesta_ticket,
                    IFNULL(nc.complemento_cesta_ticket, 30.00) as complemento_cesta_ticket,
                    IFNULL(nc.bono_transporte, 40.00) as bono_transporte,
                    IFNULL(nc.complemento_productividad, 0.00) as complemento_productividad
                FROM usuarios u 
                LEFT JOIN roles r ON u.rol_id = r.id 
                LEFT JOIN nomina_configuracion nc ON u.id = nc.usuario_id
                WHERE u.estado = 1
                ORDER BY u.nombre ASC
            ");
            $stmt->execute();
            $empleados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => ['empleados' => $empleados]];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (cargar_datos_iniciales): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', 'Error al cargar datos iniciales de empleados para nómina: ' . $e->getMessage());
            $response['message'] = 'Error al cargar los datos de los empleados.';
        }
        break;

    case 'crear_nomina':
        if (!puede('gestionar_nomina')) break;
        
        $nombre_nomina = trim($_POST['nombre_nomina'] ?? '');
        $fecha_inicio = trim($_POST['fecha_inicio'] ?? '');
        $fecha_fin = trim($_POST['fecha_fin'] ?? '');
        $tasa_cambio = floatval($_POST['tasa_cambio'] ?? 0);
        $detalles = json_decode($_POST['detalles'] ?? '[]', true);

        registrarBitacoraDetallada($conn, 'intento_crear_nomina', "Intentó crear la nómina '{$nombre_nomina}'.");

        if (empty($nombre_nomina) || empty($fecha_inicio) || empty($fecha_fin) || $tasa_cambio <= 0 || empty($detalles)) {
            $response['message'] = 'Faltan datos. Verifique el nombre, las fechas, la tasa de cambio y que haya añadido empleados.';
            registrarBitacoraDetallada($conn, 'error_validacion_nomina', "Error de validación al crear nómina '{$nombre_nomina}': Faltan datos obligatorios.");
            break;
        }

        try {
            $conn->beginTransaction();
            $stmt_nomina = $conn->prepare("INSERT INTO nominas (nombre_nomina, fecha_inicio, fecha_fin, tasa_cambio, creado_por_id, estado) VALUES (?, ?, ?, ?, ?, 'Generada')");
            $stmt_nomina->execute([$nombre_nomina, $fecha_inicio, $fecha_fin, $tasa_cambio, $_SESSION['user_id']]);
            $nomina_id = $conn->lastInsertId();

            $stmt_detalle = $conn->prepare(
                "INSERT INTO nomina_detalles (
                    nomina_id, empleado_id, sueldo_base, salario_diario_usd, cesta_ticket, cesta_ticket_diario_usd, 
                    complemento_cesta_ticket, comp_cesta_ticket_diario_usd, complemento_productividad, bono_transporte, 
                    bono_transporte_diario_usd, total_asignaciones, adelantos, dias_no_laborados, deducciones, total_deducciones, 
                    total_pagar, total_pagar_bs, observacion
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
            );

            $total_pagar_usd = 0;
            $total_pagar_bs = 0;
            foreach ($detalles as $d) {
                $stmt_detalle->execute([
                    $nomina_id, $d['empleado_id'], $d['sueldo_base'], $d['salario_diario_usd'], $d['cesta_ticket'], 
                    $d['cesta_ticket_diario_usd'], $d['complemento_cesta_ticket'], $d['comp_cesta_ticket_diario_usd'], 
                    $d['complemento_productividad'], $d['bono_transporte'], $d['bono_transporte_diario_usd'], 
                    $d['total_asignaciones'], $d['adelantos'], $d['dias_no_laborados'], $d['deducciones'], $d['total_deducciones'], 
                    $d['total_pagar'], $d['total_pagar_bs'], trim($d['observacion'] ?? '')
                ]);
                $total_pagar_usd += floatval($d['total_pagar']);
                $total_pagar_bs += floatval($d['total_pagar_bs']);
            }
            
            $conn->commit();
            
            $detalle_log = "Creó la nómina '{$nombre_nomina}' (ID: {$nomina_id}) para el período del {$fecha_inicio} al {$fecha_fin}. Tasa: {$tasa_cambio}. Incluyó a " . count($detalles) . " empleados con un total de " . number_format($total_pagar_usd, 2) . " USD / " . number_format($total_pagar_bs, 2) . " Bs.";
            registrarBitacoraDetallada($conn, 'creacion_nomina', $detalle_log);
            
            $response = ['success' => true, 'message' => 'Nómina creada exitosamente.'];
        } catch (PDOException $e) {
            $conn->rollBack();
            error_log("Nominas AJAX (crear_nomina): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', "Error crítico al guardar nómina '{$nombre_nomina}': " . $e->getMessage());
            $response['message'] = 'Error de base de datos al crear la nómina.';
        }
        break;
        
    case 'obtener_detalle_nomina':
        if (!puede('ver_nomina') && !puede('gestionar_nomina')) break;

        $nomina_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$nomina_id) {
            $response['message'] = 'ID de nómina no válido.';
            break;
        }

        try {
            $stmt_nomina = $conn->prepare("SELECT n.*, u.nombre as creado_por FROM nominas n JOIN usuarios u ON n.creado_por_id = u.id WHERE n.id = ?");
            $stmt_nomina->execute([$nomina_id]);
            $nomina = $stmt_nomina->fetch(PDO::FETCH_ASSOC);

            if (!$nomina) {
                registrarBitacoraDetallada($conn, 'consulta_fallida_nomina', "Intentó ver detalle de nómina inexistente ID: {$nomina_id}");
                $response['message'] = 'No se encontró la nómina solicitada.';
                break;
            }

            registrarBitacoraDetallada($conn, 'consulta_nomina', "Consultó el detalle de la nómina '{$nomina['nombre_nomina']}' (ID: {$nomina_id}).");

            $stmt_detalles = $conn->prepare("SELECT nd.*, u.nombre as nombre_empleado, u.cedula FROM nomina_detalles nd JOIN usuarios u ON nd.empleado_id = u.id WHERE nd.nomina_id = ? ORDER BY u.nombre ASC");
            $stmt_detalles->execute([$nomina_id]);
            $detalles = $stmt_detalles->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => ['nomina' => $nomina, 'detalles' => $detalles]];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (obtener_detalle_nomina): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', "Error al obtener detalle de nómina ID {$nomina_id}: " . $e->getMessage());
            $response['message'] = 'Error de base de datos al obtener los detalles.';
        }
        break;

    case 'marcar_como_pagada':
        if (!puede('gestionar_nomina')) break;

        $nomina_id = filter_input(INPUT_POST, 'nomina_id', FILTER_VALIDATE_INT);
        if (!$nomina_id) {
            $response['message'] = 'ID de nómina no válido.';
            break;
        }

        try {
            $nombre_nomina = obtenerNombreNomina($conn, $nomina_id);
            $stmt = $conn->prepare("UPDATE nominas SET estado = 'Pagada' WHERE id = ? AND estado = 'Generada'");
            $stmt->execute([$nomina_id]);

            if ($stmt->rowCount() > 0) {
                registrarBitacoraDetallada($conn, 'nomina_pagada', "Marcó como PAGADA la nómina '{$nombre_nomina}' (ID: {$nomina_id}). Esta acción es irreversible.");
                $response = ['success' => true, 'message' => 'Nómina actualizada a "Pagada" exitosamente.'];
            } else {
                registrarBitacoraDetallada($conn, 'operacion_fallida_nomina', "Intentó marcar como pagada la nómina '{$nombre_nomina}', pero no se pudo (ya estaba pagada o no existe).");
                $response['message'] = 'La nómina no se pudo actualizar. Es posible que ya estuviera pagada o no exista.';
            }

        } catch (PDOException $e) {
            error_log("Nominas AJAX (marcar_como_pagada): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', "Error al marcar como pagada la nómina ID {$nomina_id}: " . $e->getMessage());
            $response['message'] = 'Error de base de datos.';
        }
        break;     

    case 'listar_nominas':
        if (!puede('ver_nomina') && !puede('gestionar_nomina')) break;
        registrarBitacoraDetallada($conn, 'consulta_lista_nominas', 'Consultó el listado histórico de nóminas.');
        try {
            $stmt = $conn->prepare("SELECT n.id, n.nombre_nomina, n.fecha_inicio, n.fecha_fin, n.estado, n.fecha_creacion, u.nombre as creado_por FROM nominas n JOIN usuarios u ON n.creado_por_id = u.id ORDER BY n.fecha_creacion DESC");
            $stmt->execute();
            $nominas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $nominas];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (listar_nominas): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', 'Error al obtener listado de nóminas: ' . $e->getMessage());
            $response['message'] = 'Error al obtener el listado de nóminas.';
        }
        break;
        
    case 'obtener_datos_para_edicion':
        if (!puede('gestionar_nomina')) break;
        $nomina_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$nomina_id) {
            $response['message'] = 'ID de nómina no válido.';
            break;
        }

        try {
            $stmt_nomina = $conn->prepare("SELECT * FROM nominas WHERE id = ?");
            $stmt_nomina->execute([$nomina_id]);
            $nomina = $stmt_nomina->fetch(PDO::FETCH_ASSOC);

            if (!$nomina || $nomina['estado'] === 'Pagada') {
                $msg = $nomina ? 'ya fue pagada' : 'no existe';
                registrarBitacoraDetallada($conn, 'edicion_fallida_nomina', "Intentó editar la nómina ID {$nomina_id}, pero {$msg}.");
                $response['message'] = 'Esta nómina no se puede editar (no existe o ya fue pagada).';
                break;
            }

            registrarBitacoraDetallada($conn, 'inicio_edicion_nomina', "Cargó datos para editar la nómina '{$nomina['nombre_nomina']}' (ID: {$nomina_id}).");

            $stmt_detalles = $conn->prepare("SELECT * FROM nomina_detalles WHERE nomina_id = ?");
            $stmt_detalles->execute([$nomina_id]);
            $detalles_nomina = $stmt_detalles->fetchAll(PDO::FETCH_ASSOC);

            $stmt_empleados = $conn->prepare("
                SELECT 
                    u.id, u.nombre, u.cedula, u.salario_base,
                    IFNULL(nc.cesta_ticket, 40.00) as cesta_ticket,
                    IFNULL(nc.complemento_cesta_ticket, 30.00) as complemento_cesta_ticket,
                    IFNULL(nc.bono_transporte, 40.00) as bono_transporte,
                    IFNULL(nc.complemento_productividad, 0.00) as complemento_productividad
                FROM usuarios u 
                LEFT JOIN nomina_configuracion nc ON u.id = nc.usuario_id
                WHERE u.estado = 1 ORDER BY u.nombre ASC
            ");
            $stmt_empleados->execute();
            $empleados = $stmt_empleados->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => ['nomina' => $nomina, 'detalles_nomina' => $detalles_nomina, 'empleados' => $empleados]];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (obtener_datos_para_edicion): " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', "Error cargando datos para edición de nómina ID {$nomina_id}: " . $e->getMessage());
            $response['message'] = 'Error de base de datos al cargar datos para edición.';
        }
        break;

    case 'editar_nomina':
        if (!puede('gestionar_nomina')) break;

        $nomina_id = filter_input(INPUT_POST, 'nomina_id', FILTER_VALIDATE_INT);
        $nombre_nomina = trim($_POST['nombre_nomina'] ?? '');
        $fecha_inicio = trim($_POST['fecha_inicio'] ?? '');
        $fecha_fin = trim($_POST['fecha_fin'] ?? '');
        $tasa_cambio = floatval($_POST['tasa_cambio'] ?? 0);
        $detalles_nuevos = json_decode($_POST['detalles'] ?? '[]', true);

        if (!$nomina_id || empty($nombre_nomina) || empty($fecha_inicio) || empty($fecha_fin) || $tasa_cambio <= 0 || empty($detalles_nuevos)) {
            $response['message'] = 'Faltan datos para actualizar la nómina.';
            break;
        }

        try {
            $conn->beginTransaction();

            // --- LÓGICA DE COMPARACIÓN ---
            // 1. Obtener datos antiguos
            $stmt_anterior = $conn->prepare("SELECT * FROM nominas WHERE id = ?");
            $stmt_anterior->execute([$nomina_id]);
            $nomina_anterior = $stmt_anterior->fetch(PDO::FETCH_ASSOC);

            $stmt_detalles_ant = $conn->prepare("SELECT nd.*, u.nombre FROM nomina_detalles nd JOIN usuarios u ON nd.empleado_id = u.id WHERE nomina_id = ?");
            $stmt_detalles_ant->execute([$nomina_id]);
            $detalles_antiguos_raw = $stmt_detalles_ant->fetchAll(PDO::FETCH_ASSOC);
            $detalles_antiguos = [];
            foreach ($detalles_antiguos_raw as $d) {
                $detalles_antiguos[$d['empleado_id']] = $d;
            }

            // 2. Comparar y construir el log
            $cambios = [];
            if ($nomina_anterior['nombre_nomina'] != $nombre_nomina) $cambios[] = "Nombre: '{$nomina_anterior['nombre_nomina']}' -> '{$nombre_nomina}'";
            if ($nomina_anterior['fecha_inicio'] != $fecha_inicio) $cambios[] = "Fecha Inicio: {$nomina_anterior['fecha_inicio']} -> {$fecha_inicio}";
            if ($nomina_anterior['fecha_fin'] != $fecha_fin) $cambios[] = "Fecha Fin: {$nomina_anterior['fecha_fin']} -> {$fecha_fin}";
            if (floatval($nomina_anterior['tasa_cambio']) != $tasa_cambio) $cambios[] = "Tasa: " . number_format($nomina_anterior['tasa_cambio'], 2) . " -> " . number_format($tasa_cambio, 2);

            $empleados_nuevos = array_column($detalles_nuevos, null, 'empleado_id');
            $empleados_antiguos_ids = array_keys($detalles_antiguos);
            $empleados_nuevos_ids = array_keys($empleados_nuevos);

            $añadidos = array_diff($empleados_nuevos_ids, $empleados_antiguos_ids);
            $eliminados = array_diff($empleados_antiguos_ids, $empleados_nuevos_ids);
            $modificados = array_intersect($empleados_nuevos_ids, $empleados_antiguos_ids);
            
            if (!empty($añadidos)) {
                $nombres = [];
                foreach($añadidos as $id) $nombres[] = $empleados_nuevos[$id]['nombre_empleado'] ?? "ID {$id}";
                $cambios[] = "Empleados Añadidos: " . implode(', ', $nombres);
            }
            if (!empty($eliminados)) {
                $nombres = [];
                foreach($eliminados as $id) $nombres[] = $detalles_antiguos[$id]['nombre'];
                $cambios[] = "Empleados Eliminados: " . implode(', ', $nombres);
            }
            foreach ($modificados as $id) {
                if (floatval($detalles_antiguos[$id]['total_pagar']) != floatval($empleados_nuevos[$id]['total_pagar'])) {
                     $cambios[] = "Modificó a {$detalles_antiguos[$id]['nombre']}: (Total Pagar: " . number_format($detalles_antiguos[$id]['total_pagar'], 2) . " -> " . number_format($empleados_nuevos[$id]['total_pagar'], 2) . " USD)";
                }
            }
            // --- FIN DE LA LÓGICA DE COMPARACIÓN ---

            // Actualizar la base de datos
            $stmt_update_nomina = $conn->prepare("UPDATE nominas SET nombre_nomina = ?, fecha_inicio = ?, fecha_fin = ?, tasa_cambio = ? WHERE id = ? AND estado != 'Pagada'");
            $stmt_update_nomina->execute([$nombre_nomina, $fecha_inicio, $fecha_fin, $tasa_cambio, $nomina_id]);

            $conn->prepare("DELETE FROM nomina_detalles WHERE nomina_id = ?")->execute([$nomina_id]);

            $stmt_detalle_insert = $conn->prepare( "INSERT INTO nomina_detalles (nomina_id, empleado_id, sueldo_base, salario_diario_usd, cesta_ticket, cesta_ticket_diario_usd, complemento_cesta_ticket, comp_cesta_ticket_diario_usd, complemento_productividad, bono_transporte, bono_transporte_diario_usd, total_asignaciones, adelantos, dias_no_laborados, deducciones, total_deducciones, total_pagar, total_pagar_bs, observacion) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

            foreach ($detalles_nuevos as $d) {
                $stmt_detalle_insert->execute([$nomina_id, $d['empleado_id'], $d['sueldo_base'], $d['salario_diario_usd'], $d['cesta_ticket'], $d['cesta_ticket_diario_usd'], $d['complemento_cesta_ticket'], $d['comp_cesta_ticket_diario_usd'], $d['complemento_productividad'], $d['bono_transporte'], $d['bono_transporte_diario_usd'], $d['total_asignaciones'], $d['adelantos'], $d['dias_no_laborados'], $d['deducciones'], $d['total_deducciones'], $d['total_pagar'], $d['total_pagar_bs'], trim($d['observacion'] ?? '')]);
            }

            $conn->commit();
            
            $log_detalle = "Editó la nómina '{$nomina_anterior['nombre_nomina']}' (ID: {$nomina_id}). CAMBIOS: " . (empty($cambios) ? "Sin cambios detectados en los datos principales." : implode(' | ', $cambios));
            registrarBitacoraDetallada($conn, 'edicion_nomina', $log_detalle);

            $response = ['success' => true, 'message' => 'Nómina actualizada exitosamente.'];

        } catch (PDOException $e) {
            $conn->rollBack();
            error_log("Nominas AJAX (editar_nomina): " . $e->getMessage());
            $nombre_anterior = $nomina_anterior['nombre_nomina'] ?? "ID {$nomina_id}";
            registrarBitacoraDetallada($conn, 'error_sistema_nomina', "Error crítico editando la nómina '{$nombre_anterior}': " . $e->getMessage());
            $response['message'] = 'Error de base de datos al editar la nómina.';
        }
        break;      
}

echo json_encode($response);