<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// --- FUNCIONES AUXILIARES ---
function validarAccesoEmpleados($nivel = 'basico') {
    return $nivel === 'avanzado' ? puede('gestionar_empleados_avanzado') : 
           (puede('gestionar_empleados_basico') || puede('gestionar_empleados_avanzado'));
}

function validarDatos($datos) {
    $errores = [];
    if (empty(trim($datos['nombre'] ?? ''))) $errores[] = 'El nombre es obligatorio';
    if (empty(trim($datos['cedula'] ?? ''))) $errores[] = 'La cédula es obligatoria';
    $cedula = trim($datos['cedula'] ?? '');
    if ($cedula && !preg_match('/^[0-9V\-\s]+$/', $cedula)) $errores[] = 'Formato de cédula inválido';
    return $errores;
}

function validarCredenciales($datos, $esNuevo = false) {
    $errores = [];
    if (!empty($datos['usuario']) && empty($datos['password']) && $esNuevo) {
        $errores[] = 'Si asigna un usuario, debe proporcionar una contraseña';
    }
    if (!empty($datos['usuario']) && !preg_match('/^[a-zA-Z0-9_.\-\*]+$/', $datos['usuario'])) {
        $errores[] = 'El usuario solo puede contener letras, números, puntos, guiones bajos y asteriscos';
    }
    if (!empty($datos['password']) && strlen($datos['password']) < 6) {
        $errores[] = 'La contraseña debe tener al menos 6 caracteres';
    }
    return $errores;
}

function verificarDuplicados($conn, $cedula, $usuario = null, $empleadoId = null) {
    $errores = [];
    $cedulaLimpia = preg_replace('/[\s\-]+/', '', $cedula);
    $sql = "SELECT id, nombre FROM usuarios WHERE REPLACE(REPLACE(cedula, ' ', ''), '-', '') = ?";
    $params = [$cedulaLimpia];
    if ($empleadoId) { $sql .= " AND id != ?"; $params[] = $empleadoId; }
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    if ($result = $stmt->fetch()) $errores[] = "La cédula ya está registrada para el empleado: {$result['nombre']}";
    
    if (!empty($usuario)) {
        $sql = "SELECT id, nombre FROM usuarios WHERE usuario = ?";
        $params = [$usuario];
        if ($empleadoId) { $sql .= " AND id != ?"; $params[] = $empleadoId; }
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        if ($result = $stmt->fetch()) $errores[] = "El usuario ya está registrado para el empleado: {$result['nombre']}";
    }
    return $errores;
}

function guardarFoto($base64_string) {
    if (empty($base64_string) || strpos($base64_string, 'data:image/') === false) 
        return ['success' => false, 'message' => 'No se proporcionó una imagen válida.'];
    
    $upload_dir = __DIR__ . '/../uploads/perfiles/';
    if (!is_dir($upload_dir) && !@mkdir($upload_dir, 0775, true)) 
        return ['success' => false, 'message' => 'Error: No se pudo crear el directorio de subida.'];
    
    try {
        $data_parts = explode(',', $base64_string);
        if (count($data_parts) !== 2) return ['success' => false, 'message' => 'Formato de imagen inválido.'];
        
        $header = $data_parts[0]; $data = $data_parts[1];
        if (!preg_match('/^data:image\/(jpeg|jpg|png|gif);base64$/', $header)) 
            return ['success' => false, 'message' => 'Solo se permiten imágenes JPG, PNG, GIF.'];
        
        $decoded_data = base64_decode($data, true);
        if ($decoded_data === false) return ['success' => false, 'message' => 'Error al decodificar la imagen.'];
        
        $image_info = getimagesizefromstring($decoded_data);
        if ($image_info === false) return ['success' => false, 'message' => 'El archivo no es una imagen válida.'];
        
        $mime_to_ext = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/gif' => 'gif'];
        $ext = $mime_to_ext[$image_info['mime']] ?? 'jpg';
        $filename = uniqid('perfil_', true) . '.' . $ext;
        $destination = $upload_dir . $filename;
        
        return file_put_contents($destination, $decoded_data) ? 
               ['success' => true, 'filename' => $filename] : 
               ['success' => false, 'message' => 'No se pudo guardar la imagen en el servidor.'];
    } catch (Exception $e) {
        error_log("Error guardando imagen: " . $e->getMessage());
        return ['success' => false, 'message' => 'Error interno al procesar la imagen.'];
    }
}

function convertirFecha($fecha) {
    if (empty($fecha)) return null;
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha)) return $fecha;
    if (preg_match('/^\d{1,2}\/\d{1,2}\/\d{4}$/', $fecha)) {
        $partes = explode('/', $fecha);
        if (count($partes) === 3) return sprintf('%s-%02d-%02d', $partes[2], $partes[1], $partes[0]);
    }
    return null;
}

function procesarSedes($conn, $usuarioId, $sedes) {
    $conn->prepare("DELETE FROM usuario_sede WHERE usuario_id = ?")->execute([$usuarioId]);
    if (!empty($sedes) && is_array($sedes)) {
        $stmt_sede = $conn->prepare("INSERT INTO usuario_sede (usuario_id, sede_id) VALUES (?, ?)");
        foreach ($sedes as $sede_id) {
            $sede_id = intval($sede_id);
            if ($sede_id > 0) $stmt_sede->execute([$usuarioId, $sede_id]);
        }
    }
}

function limpiarDatos($datos) {
    $campos_texto = ['nombre', 'cedula', 'telefono', 'tipo_contrato', 'direccion', 'contacto_emergencia', 'usuario'];
    $resultado = [];
    foreach ($campos_texto as $campo) {
        $valor = isset($datos[$campo]) ? trim($datos[$campo]) : '';
        $resultado[$campo] = $valor !== '' ? $valor : null;
    }
    $resultado['salario_base'] = !empty($datos['salario_base']) ? floatval($datos['salario_base']) : null;
    $resultado['rol_id'] = !empty($datos['rol_id']) ? intval($datos['rol_id']) : null;
    $resultado['fecha_ingreso'] = convertirFecha($datos['fecha_ingreso'] ?? '');
    return $resultado;
}

// === FUNCIONES PARA BITÁCORA DETALLADA ===

function obtenerNombreRol($conn, $rol_id) {
    if (empty($rol_id)) return 'Sin rol';
    $stmt = $conn->prepare("SELECT nombre FROM roles WHERE id = ?");
    $stmt->execute([$rol_id]);
    return $stmt->fetchColumn() ?: 'Sin rol';
}

function obtenerNombresPermisos($conn, $permisos) {
    if (empty($permisos) || !is_array($permisos)) return [];
    $placeholders = str_repeat('?,', count($permisos) - 1) . '?';
    $stmt = $conn->prepare("SELECT nombre FROM permisos WHERE id IN ($placeholders)");
    $stmt->execute($permisos);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

function obtenerNombresSedes($conn, $sedes) {
    if (empty($sedes) || !is_array($sedes)) return [];
    $placeholders = str_repeat('?,', count($sedes) - 1) . '?';
    $stmt = $conn->prepare("SELECT nombre FROM sedes WHERE id IN ($placeholders)");
    $stmt->execute($sedes);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

function obtenerSedesActuales($conn, $usuario_id) {
    $stmt = $conn->prepare("SELECT sede_id FROM usuario_sede WHERE usuario_id = ?");
    $stmt->execute([$usuario_id]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

function obtenerPermisosActuales($conn, $usuario_id) {
    $stmt = $conn->prepare("SELECT permiso_id FROM usuario_permiso WHERE usuario_id = ?");
    $stmt->execute([$usuario_id]);
    return array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
}

function registrarBitacoraDetallada($conn, $accion, $detalle_especifico) {
    if (!isset($_SESSION['user_id'])) return;
    
    $usuario_admin = $_SESSION['user_nombre'] ?? 'Usuario Desconocido';
    $detalle_completo = "ADMIN: {$usuario_admin} - {$detalle_especifico}";
    
    registrar_accion($accion, $detalle_completo);
}

function compararYRegistrarCambios($conn, $datos_anteriores, $datos_nuevos, $empleado_nombre) {
    $cambios = [];
    $campos_mapeo = [
        'nombre' => 'Nombre completo',
        'cedula' => 'Cédula de identidad', 
        'telefono' => 'Número telefónico',
        'fecha_ingreso' => 'Fecha de ingreso',
        'tipo_contrato' => 'Tipo de contrato',
        'salario_base' => 'Salario base',
        'direccion' => 'Dirección de habitación',
        'contacto_emergencia' => 'Contacto de emergencia',
        'usuario' => 'Usuario de acceso',
        'rol_id' => 'Rol asignado'
    ];
    
    foreach ($campos_mapeo as $campo => $descripcion) {
        $anterior = $datos_anteriores[$campo] ?? null;
        $nuevo = $datos_nuevos[$campo] ?? null;
        
        if ($campo === 'rol_id') {
            $anterior = obtenerNombreRol($conn, $anterior);
            $nuevo = obtenerNombreRol($conn, $nuevo);
        }
        
        if ($anterior != $nuevo) {
            $anterior_texto = $anterior ?: '[Vacío]';
            $nuevo_texto = $nuevo ?: '[Vacío]';
            $cambios[] = "{$descripcion}: '{$anterior_texto}' → '{$nuevo_texto}'";
        }
    }
    
    if (!empty($cambios)) {
        $detalle = "Modificó datos del empleado '{$empleado_nombre}' (ID: {$datos_anteriores['id']}). CAMBIOS: " . implode(' | ', $cambios);
        registrarBitacoraDetallada($conn, 'modificar_empleado', $detalle);
    }
    
    return $cambios;
}

// --- CONTROLADORES ---
switch ($action) {
    case 'cargar_dashboard_empleados':
        if (!validarAccesoEmpleados()) { $response['message'] = 'Acceso denegado.'; break; }
        
        // Registrar acceso al dashboard
        registrarBitacoraDetallada($conn, 'acceso_modulo', 'Accedió al módulo de Gestión de Empleados - Dashboard');
        
        try {
            $empleados = $conn->query("
                SELECT u.id, u.usuario, u.nombre, u.cedula, u.estado, u.foto_perfil, 
                       r.nombre as rol_nombre, 
                       GROUP_CONCAT(s.nombre SEPARATOR ', ') as sedes_nombres
                FROM usuarios u 
                LEFT JOIN roles r ON u.rol_id = r.id 
                LEFT JOIN usuario_sede us ON u.id = us.usuario_id
                LEFT JOIN sedes s ON us.sede_id = s.id
                GROUP BY u.id ORDER BY u.nombre ASC
            ")->fetchAll(PDO::FETCH_ASSOC);
            $roles = $conn->query("SELECT id, nombre FROM roles WHERE clave != 'superadmin' ORDER BY nombre ASC")->fetchAll(PDO::FETCH_ASSOC);
            $sedes = $conn->query("SELECT id, nombre FROM sedes ORDER BY nombre ASC")->fetchAll(PDO::FETCH_ASSOC);
            
            $total_empleados = count($empleados);
            registrarBitacoraDetallada($conn, 'consulta_datos', "Consultó listado completo de empleados. Total encontrados: {$total_empleados} empleados");
            
            $response = ['success' => true, 'data' => [
                'empleados' => $empleados, 'roles' => $roles, 'sedes' => $sedes, 'total_empleados' => $total_empleados
            ]];
        } catch (Exception $e) {
            error_log("Error cargando dashboard: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', 'Error al cargar dashboard de empleados: ' . $e->getMessage());
            $response['message'] = 'Error al cargar los datos del dashboard.';
        }
        break;

    case 'agregar_empleado':
        if (!validarAccesoEmpleados()) { $response['message'] = 'Acceso denegado.'; break; }
        $datos = limpiarDatos($_POST);
        $errores = validarDatos($datos);
        $puedeAcceso = validarAccesoEmpleados('avanzado');
        $usuario = null;
        
        // Registrar intento de creación
        registrarBitacoraDetallada($conn, 'intento_crear_empleado', "Intentando crear nuevo empleado: '{$datos['nombre']}' con cédula '{$datos['cedula']}'");
        
        if (!$puedeAcceso) {
            $datos['usuario'] = null;
        } else if (!empty($datos['usuario']) || !empty($_POST['password'])) {
            $credenciales = $datos;
            $credenciales['password'] = $_POST['password'] ?? '';
            $errores = array_merge($errores, validarCredenciales($credenciales, true));
            if (!empty($datos['usuario'])) $usuario = $datos['usuario'];
        }
        
        $errores = array_merge($errores, verificarDuplicados($conn, $datos['cedula'], $usuario));
        if (!empty($errores)) { 
            registrarBitacoraDetallada($conn, 'error_validacion', "Error al crear empleado '{$datos['nombre']}': " . implode(', ', $errores));
            $response['message'] = implode('<br>', $errores); 
            break; 
        }
        
        try {
            $conn->beginTransaction();
            $foto = null;
            
            if (!empty($_POST['foto_perfil_base64'])) {
                $upload = guardarFoto($_POST['foto_perfil_base64']);
                if ($upload['success']) {
                    $foto = $upload['filename'];
                    registrarBitacoraDetallada($conn, 'subir_archivo', "Subió foto de perfil para empleado '{$datos['nombre']}': {$foto}");
                } else { 
                    $conn->rollBack(); 
                    registrarBitacoraDetallada($conn, 'error_archivo', "Error subiendo foto para empleado '{$datos['nombre']}': {$upload['message']}");
                    $response['message'] = $upload['message']; 
                    break; 
                }
            }
            
            $estado = 0;
            $password_hash = null;
            if ($puedeAcceso && !empty($datos['usuario']) && !empty($_POST['password'])) {
                $estado = 1;
                $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
            }
            
            $stmt = $conn->prepare(
                "INSERT INTO usuarios (nombre, cedula, telefono, fecha_ingreso, tipo_contrato, salario_base, 
                 direccion, contacto_emergencia, foto_perfil, estado, usuario, password, rol_id) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
            );
            $stmt->execute([
                $datos['nombre'], $datos['cedula'], $datos['telefono'], $datos['fecha_ingreso'],
                $datos['tipo_contrato'], $datos['salario_base'], $datos['direccion'], 
                $datos['contacto_emergencia'], $foto, $estado, $datos['usuario'], 
                $password_hash, $datos['rol_id']
            ]);
            
            $nuevo_id = $conn->lastInsertId();
            
            // Procesar sedes
            procesarSedes($conn, $nuevo_id, $_POST['sedes'] ?? []);
            
            $conn->commit();
            
            // Registrar creación exitosa con detalles completos
            $rol_nombre = obtenerNombreRol($conn, $datos['rol_id']);
            $sedes_nombres = obtenerNombresSedes($conn, $_POST['sedes'] ?? []);
            
            $detalle_creacion = "Creó empleado exitosamente: ID#{$nuevo_id} '{$datos['nombre']}' | Cédula: '{$datos['cedula']}' | Rol: '{$rol_nombre}'";
            
            if (!empty($datos['usuario'])) {
                $detalle_creacion .= " | Usuario: '{$datos['usuario']}' (Acceso: " . ($estado ? 'HABILITADO' : 'DESHABILITADO') . ")";
            }
            
            if (!empty($datos['telefono'])) $detalle_creacion .= " | Teléfono: '{$datos['telefono']}'";
            if (!empty($datos['salario_base'])) $detalle_creacion .= " | Salario: Bs. " . number_format($datos['salario_base'], 2);
            if (!empty($sedes_nombres)) $detalle_creacion .= " | Sedes: " . implode(', ', $sedes_nombres);
            if (!empty($datos['tipo_contrato'])) $detalle_creacion .= " | Contrato: '{$datos['tipo_contrato']}'";
            if (!empty($datos['fecha_ingreso'])) $detalle_creacion .= " | Ingreso: {$datos['fecha_ingreso']}";
            
            registrarBitacoraDetallada($conn, 'crear_empleado', $detalle_creacion);
            
            $mensaje = 'Empleado registrado exitosamente.';
            if (!$puedeAcceso && (!empty($_POST['usuario']) || !empty($_POST['password']))) {
                $mensaje .= ' Nota: No se creó acceso al sistema (permisos insuficientes).';
                registrarBitacoraDetallada($conn, 'limitacion_permisos', "No pudo crear acceso de usuario para '{$datos['nombre']}' por permisos insuficientes");
            }
            $response = ['success' => true, 'message' => $mensaje];
            
        } catch (Exception $e) {
            $conn->rollBack();
            error_log("Error agregando empleado: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', "Error crítico creando empleado '{$datos['nombre']}': " . $e->getMessage());
            $response['message'] = 'Error al registrar el empleado.';
        }
        break;

    case 'obtener_empleado':
        if (!validarAccesoEmpleados()) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$id) { $response['message'] = 'ID de empleado no válido.'; break; }

        try {
            $stmt = $conn->prepare("SELECT u.*, r.nombre as rol_nombre FROM usuarios u LEFT JOIN roles r ON u.rol_id = r.id WHERE u.id = ?");
            $stmt->execute([$id]);
            $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($usuario) {
                // Registrar consulta específica
                registrarBitacoraDetallada($conn, 'consultar_empleado', "Consultó detalles del empleado: ID#{$id} '{$usuario['nombre']}'");
                
                // CRÍTICO: Nunca devolver la contraseña hasheada
                unset($usuario['password']);
                
                $stmt_sedes = $conn->prepare("SELECT sede_id FROM usuario_sede WHERE usuario_id = ?");
                $stmt_sedes->execute([$id]);
                $usuario['sedes'] = $stmt_sedes->fetchAll(PDO::FETCH_COLUMN);
                
                $usuario['permisos_especiales'] = [];
                $usuario['permisos_del_rol'] = [];
                
                if (validarAccesoEmpleados('avanzado')) {
                    $stmt_esp = $conn->prepare("SELECT permiso_id FROM usuario_permiso WHERE usuario_id = ?");
                    $stmt_esp->execute([$id]);
                    $usuario['permisos_especiales'] = array_map('intval', $stmt_esp->fetchAll(PDO::FETCH_COLUMN));

                    if (!empty($usuario['rol_id'])) {
                        $stmt_rol = $conn->prepare("SELECT permiso_id FROM rol_permiso WHERE rol_id = ?");
                        $stmt_rol->execute([$usuario['rol_id']]);
                        $usuario['permisos_del_rol'] = array_map('intval', $stmt_rol->fetchAll(PDO::FETCH_COLUMN));
                    }
                }
                
                $response = ['success' => true, 'data' => $usuario];
            } else {
                registrarBitacoraDetallada($conn, 'consulta_fallida', "Intentó consultar empleado inexistente con ID: {$id}");
                $response['message'] = 'Empleado no encontrado.';
            }
        } catch (Exception $e) {
            error_log("Error obteniendo empleado: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', "Error consultando empleado ID {$id}: " . $e->getMessage());
            $response['message'] = 'Error al obtener los datos del empleado.';
        }
        break;

    case 'editar_empleado':
        if (!validarAccesoEmpleados()) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
        if (!$id || empty($_POST['nombre']) || empty($_POST['cedula'])) {
            $response['message'] = 'Datos obligatorios faltantes.'; break;
        }

        try {
            $conn->beginTransaction();
            $puedeAcceso = validarAccesoEmpleados('avanzado');
            
            // Obtener datos actuales COMPLETOS para comparar
            $stmt_current = $conn->prepare("SELECT * FROM usuarios WHERE id = ?");
            $stmt_current->execute([$id]);
            $datos_anteriores = $stmt_current->fetch(PDO::FETCH_ASSOC);
            if (!$datos_anteriores) { 
                $conn->rollBack(); 
                registrarBitacoraDetallada($conn, 'error_validacion', "Intentó editar empleado inexistente con ID: {$id}");
                $response['message'] = 'Empleado no encontrado.'; 
                break; 
            }

            // Registrar inicio de edición
            registrarBitacoraDetallada($conn, 'iniciar_edicion', "Iniciando edición del empleado: ID#{$id} '{$datos_anteriores['nombre']}'");
            
            // Verificar duplicados considerando permisos
            $usuario = $puedeAcceso ? ($_POST['usuario'] ?? null) : null;
            $errores = verificarDuplicados($conn, $_POST['cedula'], $usuario, $id);
            if (!empty($errores)) { 
                $conn->rollBack(); 
                registrarBitacoraDetallada($conn, 'error_validacion', "Error editando empleado '{$datos_anteriores['nombre']}': " . implode(', ', $errores));
                $response['message'] = implode('<br>', $errores); 
                break; 
            }

            $updates = []; $params = [];
            
            // Campos básicos que todos pueden editar
            $campos_basicos = ['nombre', 'cedula', 'telefono', 'tipo_contrato', 'salario_base', 'direccion', 'contacto_emergencia'];
            
            // Campos avanzados solo para usuarios con permisos
            if ($puedeAcceso) {
                $campos_basicos = array_merge($campos_basicos, ['usuario', 'rol_id']);
            }
            
            foreach($campos_basicos as $campo) {
                $valor = isset($_POST[$campo]) && trim($_POST[$campo]) !== '' ? trim($_POST[$campo]) : null;
                if ($campo === 'salario_base' && $valor !== null) $valor = floatval($valor);
                if ($campo === 'rol_id' && $valor !== null) $valor = intval($valor);
                if ($valor != $datos_anteriores[$campo]) { $updates[] = "{$campo} = ?"; $params[] = $valor; }
            }
            
            $fecha = convertirFecha($_POST['fecha_ingreso'] ?? '');
            if ($fecha != $datos_anteriores['fecha_ingreso']) { $updates[] = "fecha_ingreso = ?"; $params[] = $fecha; }
            
            // Solo usuarios avanzados pueden cambiar contraseñas
            $cambio_password = false;
            if ($puedeAcceso && !empty($_POST['password'])) {
                $updates[] = "password = ?"; 
                $params[] = password_hash($_POST['password'], PASSWORD_DEFAULT);
                $cambio_password = true;
            }
            
            // Manejo de foto
            $cambio_foto = false;
            if (!empty($_POST['foto_perfil_base64'])) {
                $upload = guardarFoto($_POST['foto_perfil_base64']);
                if ($upload['success']) {
                    $updates[] = "foto_perfil = ?"; $params[] = $upload['filename'];
                    if ($datos_anteriores['foto_perfil']) {
                        $old_photo = __DIR__ . '/../uploads/perfiles/' . $datos_anteriores['foto_perfil'];
                        if (file_exists($old_photo)) @unlink($old_photo);
                    }
                    $cambio_foto = true;
                    registrarBitacoraDetallada($conn, 'cambiar_foto', "Cambió foto de perfil del empleado '{$datos_anteriores['nombre']}': {$upload['filename']}");
                } else { 
                    $conn->rollBack(); 
                    registrarBitacoraDetallada($conn, 'error_archivo', "Error cambiando foto del empleado '{$datos_anteriores['nombre']}': {$upload['message']}");
                    $response['message'] = $upload['message']; 
                    break; 
                }
            }
            
            // Aplicar cambios si existen
            if (!empty($updates)) {
                $sql = "UPDATE usuarios SET " . implode(', ', $updates) . " WHERE id = ?";
                $params[] = $id;
                $conn->prepare($sql)->execute($params);
                
                // Crear array con datos nuevos para comparación detallada
                $datos_nuevos = $datos_anteriores;
                foreach ($_POST as $key => $value) {
                    if (in_array($key, $campos_basicos)) {
                        $datos_nuevos[$key] = trim($value) !== '' ? trim($value) : null;
                        if ($key === 'salario_base' && $datos_nuevos[$key] !== null) $datos_nuevos[$key] = floatval($datos_nuevos[$key]);
                        if ($key === 'rol_id' && $datos_nuevos[$key] !== null) $datos_nuevos[$key] = intval($datos_nuevos[$key]);
                    }
                }
                if ($fecha !== null) $datos_nuevos['fecha_ingreso'] = $fecha;
                
                // Registrar cambios detallados
                compararYRegistrarCambios($conn, $datos_anteriores, $datos_nuevos, $datos_anteriores['nombre']);
            }
            
            // Cambios especiales
            if ($cambio_password) {
                registrarBitacoraDetallada($conn, 'cambiar_password', "Cambió contraseña de acceso del empleado '{$datos_anteriores['nombre']}'");
            }
            
            // Procesar cambios de sedes
            $sedes_anteriores = obtenerSedesActuales($conn, $id);
            $sedes_nuevas = $_POST['sedes'] ?? [];
            
            if (array_diff($sedes_anteriores, $sedes_nuevas) || array_diff($sedes_nuevas, $sedes_anteriores)) {
                $nombres_anteriores = obtenerNombresSedes($conn, $sedes_anteriores);
                $nombres_nuevas = obtenerNombresSedes($conn, $sedes_nuevas);
                
                $anterior_texto = !empty($nombres_anteriores) ? implode(', ', $nombres_anteriores) : '[Sin sedes]';
                $nuevo_texto = !empty($nombres_nuevas) ? implode(', ', $nombres_nuevas) : '[Sin sedes]';
                
                registrarBitacoraDetallada($conn, 'cambiar_sedes', "Modificó sedes del empleado '{$datos_anteriores['nombre']}': '{$anterior_texto}' → '{$nuevo_texto}'");
            }
            
            procesarSedes($conn, $id, $sedes_nuevas);
            $conn->commit();
            
            registrarBitacoraDetallada($conn, 'completar_edicion', "Completó exitosamente la edición del empleado: ID#{$id} '{$datos_anteriores['nombre']}'");
            $response = ['success' => true, 'message' => 'Empleado actualizado exitosamente.'];
            
        } catch (Exception $e) {
            $conn->rollBack();
            error_log("Error editando empleado: " . $e->getMessage());
            $nombre_empleado = $datos_anteriores['nombre'] ?? "ID {$id}";
            registrarBitacoraDetallada($conn, 'error_sistema', "Error crítico editando empleado '{$nombre_empleado}': " . $e->getMessage());
            $response['message'] = 'Error al actualizar el empleado.';
        }
        break;

    case 'guardar_permisos_usuario':
        if (!validarAccesoEmpleados('avanzado')) { $response['message'] = 'Acceso denegado.'; break; }
        $usuario_id = filter_input(INPUT_POST, 'usuario_id', FILTER_VALIDATE_INT);
        $permisos_nuevos = $_POST['permisos'] ?? [];
        if (!$usuario_id) { $response['message'] = 'ID de usuario no válido.'; break; }
        
        try {
            $conn->beginTransaction();
            $stmt_check = $conn->prepare("SELECT nombre FROM usuarios WHERE id = ?");
            $stmt_check->execute([$usuario_id]);
            $nombre_empleado = $stmt_check->fetchColumn();
            if (!$nombre_empleado) { 
                $conn->rollBack(); 
                registrarBitacoraDetallada($conn, 'error_validacion', "Intentó modificar permisos de usuario inexistente con ID: {$usuario_id}");
                $response['message'] = 'Usuario no encontrado.'; 
                break; 
            }
            
            // Obtener permisos actuales para comparar
            $permisos_anteriores = obtenerPermisosActuales($conn, $usuario_id);
            
            registrarBitacoraDetallada($conn, 'iniciar_permisos', "Iniciando modificación de permisos especiales para empleado: '{$nombre_empleado}' (ID#{$usuario_id})");
            
            $conn->prepare("DELETE FROM usuario_permiso WHERE usuario_id = ?")->execute([$usuario_id]);
            
            if (!empty($permisos_nuevos) && is_array($permisos_nuevos)) {
                $stmt_permiso = $conn->prepare("INSERT INTO usuario_permiso (usuario_id, permiso_id) VALUES (?, ?)");
                foreach ($permisos_nuevos as $permiso_id) {
                    $permiso_id = intval($permiso_id);
                    if ($permiso_id > 0) $stmt_permiso->execute([$usuario_id, $permiso_id]);
                }
            }
            
            // Comparar y registrar cambios detallados SOLO si no hay errores
            try {
                $nombres_anteriores = obtenerNombresPermisos($conn, $permisos_anteriores);
                $nombres_nuevos = obtenerNombresPermisos($conn, array_map('intval', $permisos_nuevos));
                
                $anterior_texto = !empty($nombres_anteriores) ? implode(', ', $nombres_anteriores) : '[Sin permisos especiales]';
                $nuevo_texto = !empty($nombres_nuevos) ? implode(', ', $nombres_nuevos) : '[Sin permisos especiales]';
                
                $detalle_permisos = "Modificó permisos especiales del empleado '{$nombre_empleado}' (ID#{$usuario_id}): '{$anterior_texto}' → '{$nuevo_texto}'";
                
                // Detallar qué permisos se agregaron y cuáles se quitaron
                $permisos_agregados = array_diff($nombres_nuevos, $nombres_anteriores);
                $permisos_removidos = array_diff($nombres_anteriores, $nombres_nuevos);
                
                if (!empty($permisos_agregados)) {
                    $detalle_permisos .= " | AGREGÓ: " . implode(', ', $permisos_agregados);
                }
                if (!empty($permisos_removidos)) {
                    $detalle_permisos .= " | REMOVIÓ: " . implode(', ', $permisos_removidos);
                }
                
                registrarBitacoraDetallada($conn, 'modificar_permisos', $detalle_permisos);
            } catch (Exception $log_error) {
                // Si falla el log detallado, registrar uno simple
                registrarBitacoraDetallada($conn, 'modificar_permisos', "Modificó permisos especiales del empleado '{$nombre_empleado}' (ID#{$usuario_id})");
            }
            
            $conn->commit();
            $response = ['success' => true, 'message' => 'Permisos especiales actualizados correctamente.'];
            
        } catch (Exception $e) {
            $conn->rollBack();
            error_log("Error guardando permisos: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', "Error crítico modificando permisos del empleado ID {$usuario_id}: " . $e->getMessage());
            $response['message'] = 'Error al guardar los permisos.';
        }
        break;

    case 'cambiar_estado':
        if (!validarAccesoEmpleados('avanzado')) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
        if (!$id) { $response['message'] = 'ID de empleado no válido.'; break; }
        
        try {
            // Obtener estado actual y datos del empleado
            $stmt_actual = $conn->prepare("SELECT nombre, estado, usuario FROM usuarios WHERE id = ?");
            $stmt_actual->execute([$id]);
            $datos_empleado = $stmt_actual->fetch(PDO::FETCH_ASSOC);
            
            if (!$datos_empleado) {
                registrarBitacoraDetallada($conn, 'error_validacion', "Intentó cambiar estado de empleado inexistente con ID: {$id}");
                $response['message'] = 'Empleado no encontrado.';
                break;
            }
            
            $estado_anterior = $datos_empleado['estado'];
            $nuevo_estado = 1 - $estado_anterior; // Cambiar entre 0 y 1
            $nombre_empleado = $datos_empleado['nombre'];
            $usuario_acceso = $datos_empleado['usuario'];
            
            $stmt = $conn->prepare("UPDATE usuarios SET estado = ? WHERE id = ?");
            $stmt->execute([$nuevo_estado, $id]);
            
            if ($stmt->rowCount() > 0) {
                $estado_texto_anterior = $estado_anterior ? 'HABILITADO' : 'DESHABILITADO';
                $estado_texto_nuevo = $nuevo_estado ? 'HABILITADO' : 'DESHABILITADO';
                
                $detalle_estado = "Cambió estado de acceso del empleado '{$nombre_empleado}' (ID#{$id}): {$estado_texto_anterior} → {$estado_texto_nuevo}";
                
                if (!empty($usuario_acceso)) {
                    $detalle_estado .= " | Usuario afectado: '{$usuario_acceso}'";
                }
                
                registrarBitacoraDetallada($conn, 'cambiar_estado_acceso', $detalle_estado);
                
                $response = ['success' => true, 'message' => 'Estado de acceso actualizado correctamente.'];
            } else {
                registrarBitacoraDetallada($conn, 'error_operacion', "No se pudo cambiar el estado del empleado '{$nombre_empleado}' (ID#{$id}) - Sin cambios realizados");
                $response = ['success' => false, 'message' => 'No se pudo actualizar el estado. Empleado no encontrado.'];
            }
        } catch (Exception $e) {
            error_log("Error cambiando estado: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', "Error crítico cambiando estado del empleado ID {$id}: " . $e->getMessage());
            $response['message'] = 'Error al cambiar el estado del empleado.';
        }
        break;

    case 'eliminar_empleado':
        if (!validarAccesoEmpleados('avanzado')) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
        if (!$id) { $response['message'] = 'ID de empleado no válido.'; break; }
        
        try {
            $conn->beginTransaction();
            
            // Obtener datos completos del empleado antes de eliminar
            $stmt_datos = $conn->prepare("
                SELECT u.nombre, u.cedula, u.usuario, u.foto_perfil, r.nombre as rol_nombre,
                       GROUP_CONCAT(s.nombre SEPARATOR ', ') as sedes_nombres
                FROM usuarios u 
                LEFT JOIN roles r ON u.rol_id = r.id
                LEFT JOIN usuario_sede us ON u.id = us.usuario_id
                LEFT JOIN sedes s ON us.sede_id = s.id
                WHERE u.id = ?
                GROUP BY u.id
            ");
            $stmt_datos->execute([$id]);
            $datos_empleado = $stmt_datos->fetch(PDO::FETCH_ASSOC);
            
            if (!$datos_empleado) {
                $conn->rollBack();
                registrarBitacoraDetallada($conn, 'error_validacion', "Intentó eliminar empleado inexistente con ID: {$id}");
                $response['message'] = 'Empleado no encontrado.';
                break;
            }
            
            $nombre_empleado = $datos_empleado['nombre'];
            $cedula = $datos_empleado['cedula'];
            $usuario_acceso = $datos_empleado['usuario'];
            $rol_nombre = $datos_empleado['rol_nombre'] ?? 'Sin rol';
            $sedes_nombres = $datos_empleado['sedes_nombres'] ?? 'Sin sedes';
            $foto_perfil = $datos_empleado['foto_perfil'];
            
            registrarBitacoraDetallada($conn, 'iniciar_eliminacion', "Iniciando eliminación del empleado: ID#{$id} '{$nombre_empleado}' - Cédula: '{$cedula}'");
            
            // Eliminar relaciones
            $conn->prepare("DELETE FROM usuario_sede WHERE usuario_id = ?")->execute([$id]);
            $conn->prepare("DELETE FROM usuario_permiso WHERE usuario_id = ?")->execute([$id]);
            
            // Eliminar el empleado
            $stmt_delete = $conn->prepare("DELETE FROM usuarios WHERE id = ?");
            $stmt_delete->execute([$id]);
            
            if ($stmt_delete->rowCount() > 0) {
                // Eliminar foto si existe
                if ($foto_perfil) {
                    $foto_path = __DIR__ . '/../uploads/perfiles/' . $foto_perfil;
                    if (file_exists($foto_path)) {
                        @unlink($foto_path);
                        registrarBitacoraDetallada($conn, 'eliminar_archivo', "Eliminó foto de perfil del empleado '{$nombre_empleado}': {$foto_perfil}");
                    }
                }
                
                $detalle_eliminacion = "ELIMINÓ EMPLEADO COMPLETAMENTE: ID#{$id} '{$nombre_empleado}' | Cédula: '{$cedula}' | Rol: '{$rol_nombre}' | Sedes: {$sedes_nombres}";
                
                if ($usuario_acceso) {
                    $detalle_eliminacion .= " | Usuario de acceso: '{$usuario_acceso}' (TAMBIÉN ELIMINADO)";
                }
                
                registrarBitacoraDetallada($conn, 'eliminar_empleado', $detalle_eliminacion);
                
                $conn->commit();
                $response = ['success' => true, 'message' => 'Empleado eliminado exitosamente.'];
            } else {
                $conn->rollBack();
                registrarBitacoraDetallada($conn, 'error_operacion', "No se pudo eliminar el empleado '{$nombre_empleado}' (ID#{$id}) - Sin cambios realizados");
                $response = ['success' => false, 'message' => 'No se pudo eliminar el empleado.'];
            }
            
        } catch (Exception $e) {
            $conn->rollBack();
            error_log("Error eliminando empleado: " . $e->getMessage());
            $nombre_empleado = isset($datos_empleado) ? $datos_empleado['nombre'] : "ID {$id}";
            registrarBitacoraDetallada($conn, 'error_sistema', "Error crítico eliminando empleado '{$nombre_empleado}': " . $e->getMessage());
            $response['message'] = 'Error al eliminar el empleado.';
        }
        break;

    case 'buscar_empleados':
        if (!validarAccesoEmpleados()) { $response['message'] = 'Acceso denegado.'; break; }
        
        $termino = trim($_GET['termino'] ?? $_POST['termino'] ?? '');
        $filtro_rol = $_GET['filtro_rol'] ?? $_POST['filtro_rol'] ?? '';
        $filtro_estado = $_GET['filtro_estado'] ?? $_POST['filtro_estado'] ?? '';
        
        if (empty($termino) && empty($filtro_rol) && empty($filtro_estado)) {
            registrarBitacoraDetallada($conn, 'busqueda_empleados', 'Realizó búsqueda vacía de empleados');
            $response['message'] = 'Debe especificar al menos un criterio de búsqueda.';
            break;
        }
        
        try {
            $sql = "
                SELECT u.id, u.usuario, u.nombre, u.cedula, u.estado, u.foto_perfil, 
                       r.nombre as rol_nombre, 
                       GROUP_CONCAT(s.nombre SEPARATOR ', ') as sedes_nombres
                FROM usuarios u 
                LEFT JOIN roles r ON u.rol_id = r.id 
                LEFT JOIN usuario_sede us ON u.id = us.usuario_id
                LEFT JOIN sedes s ON us.sede_id = s.id
                WHERE 1=1
            ";
            $params = [];
            
            $criterios_busqueda = [];
            
            if (!empty($termino)) {
                $sql .= " AND (u.nombre LIKE ? OR u.cedula LIKE ? OR u.usuario LIKE ?)";
                $termino_like = "%{$termino}%";
                $params[] = $termino_like;
                $params[] = $termino_like;
                $params[] = $termino_like;
                $criterios_busqueda[] = "Término: '{$termino}'";
            }
            
            if (!empty($filtro_rol)) {
                $sql .= " AND u.rol_id = ?";
                $params[] = intval($filtro_rol);
                $stmt_rol = $conn->prepare("SELECT nombre FROM roles WHERE id = ?");
                $stmt_rol->execute([intval($filtro_rol)]);
                $nombre_rol = $stmt_rol->fetchColumn();
                $criterios_busqueda[] = "Rol: '{$nombre_rol}'";
            }
            
            if ($filtro_estado !== '') {
                $sql .= " AND u.estado = ?";
                $params[] = intval($filtro_estado);
                $estado_texto = intval($filtro_estado) ? 'Habilitados' : 'Deshabilitados';
                $criterios_busqueda[] = "Estado: {$estado_texto}";
            }
            
            $sql .= " GROUP BY u.id ORDER BY u.nombre ASC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $criterios_texto = implode(' | ', $criterios_busqueda);
            $total_encontrados = count($resultados);
            
            registrarBitacoraDetallada($conn, 'busqueda_empleados', "Realizó búsqueda de empleados con criterios: {$criterios_texto} | Resultados encontrados: {$total_encontrados}");
            
            $response = [
                'success' => true, 
                'data' => $resultados,
                'total' => $total_encontrados,
                'criterios' => $criterios_texto
            ];
            
        } catch (Exception $e) {
            error_log("Error en búsqueda: " . $e->getMessage());
            registrarBitacoraDetallada($conn, 'error_sistema', "Error en búsqueda de empleados: " . $e->getMessage());
            $response['message'] = 'Error al realizar la búsqueda.';
        }
        break;

    default:
        registrarBitacoraDetallada($conn, 'accion_invalida', "Intentó ejecutar acción no válida: '{$action}'");
        $response['message'] = 'Acción no reconocida.';
        break;
}

echo json_encode($response);