<?php
define('IS_AJAX', true);
// ESTABLECEMOS LA ZONA HORARIA DE VENEZUELA PARA TODOS LOS REGISTROS
date_default_timezone_set('America/Caracas');

require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida o error desconocido.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

// Verificación de Token CSRF para todas las peticiones POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF). Por favor, recargue la página.';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// OBTENER ROL Y SEDES PERMITIDAS DEL USUARIO LOGUEADO
try {
    $stmt_usuario = $conn->prepare("SELECT rol_id FROM usuarios WHERE id = ?");
    $stmt_usuario->execute([$usuario_id]);
    $rol_id_usuario = $stmt_usuario->fetchColumn();

    $es_superadmin = ($rol_id_usuario == 1);

    $sedes_permitidas = [];
    if (!$es_superadmin) {
        $stmt_sedes = $conn->prepare("SELECT sede_id FROM usuario_sede WHERE usuario_id = ?");
        $stmt_sedes->execute([$usuario_id]);
        $sedes_permitidas = $stmt_sedes->fetchAll(PDO::FETCH_COLUMN);
    }
} catch (PDOException $e) {
    $response['message'] = 'Error crítico al verificar permisos de sede: ' . $e->getMessage();
    echo json_encode($response);
    exit();
}

function obtenerNombreUsuario($conn, $user_id) {
    $stmt = $conn->prepare("SELECT nombre FROM usuarios WHERE id = ?");
    $stmt->execute([$user_id]);
    return $stmt->fetchColumn() ?? 'Usuario Desconocido';
}

switch ($action) {
    case 'cargar_afiliaciones':
        if (!puede('ver_afiliaciones')) {
            $response['message'] = 'Acceso denegado para ver afiliaciones.';
            break;
        }
        try {
            $search = $_GET['search'] ?? '';
            // MODIFICADO: Se añade una subconsulta para obtener la última fecha de pago
            $sql = "SELECT a.id, a.numero_contrato, a.nombre_titular, a.apellido_titular, a.cedula_titular, a.fecha_vencimiento, a.estado, a.nombre_empresa, a.tipo_contrato,
                            (SELECT COUNT(*) FROM afiliados_miembros am WHERE am.afiliacion_id = a.id) as total_miembros,
                            (SELECT MAX(hp.fecha_pago) FROM historial_pagos hp WHERE hp.afiliacion_id = a.id) as ultima_fecha_pago
                    FROM afiliaciones a";
            
            $where_clauses = [];
            $params = [];

            if (!$es_superadmin) {
                if (empty($sedes_permitidas)) {
                    $response = ['success' => true, 'data' => []];
                    echo json_encode($response);
                    exit();
                }
                $placeholders = implode(',', array_fill(0, count($sedes_permitidas), '?'));
                $where_clauses[] = "a.sede_id IN ($placeholders)";
                $params = array_merge($params, $sedes_permitidas);
            }
            
            if (!empty($search)) {
                $where_clauses[] = "(a.numero_contrato LIKE ? OR CONCAT(a.nombre_titular, ' ', a.apellido_titular) LIKE ? OR a.cedula_titular LIKE ? OR a.nombre_empresa LIKE ?)";
                $searchTerm = "%{$search}%";
                array_push($params, $searchTerm, $searchTerm, $searchTerm, $searchTerm);
            }

            if (!empty($where_clauses)) {
                $sql .= " WHERE " . implode(' AND ', $where_clauses);
            }

            $sql .= " ORDER BY a.fecha_creacion DESC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $afiliaciones = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($afiliaciones as &$a) {
                if ($a['estado'] === 'Suspendido' || $a['estado'] === 'Cancelado') continue;
                $hoy = new DateTime();
                $vencimiento = !empty($a['fecha_vencimiento']) ? new DateTime($a['fecha_vencimiento']) : null;
                if (!$vencimiento) {
                    $a['estado'] = 'Pendiente';
                } else if ($hoy > $vencimiento) {
                    $dias_vencido = $hoy->diff($vencimiento)->days;
                    $a['estado'] = ($dias_vencido > 10) ? 'Moroso' : 'Vencido';
                } else {
                    $dias_para_vencer = $hoy->diff($vencimiento)->days;
                    $a['estado'] = ($dias_para_vencer <= 5) ? 'Próximo a Vencer' : 'Activo';
                }
            }
            $response = ['success' => true, 'data' => $afiliaciones];
        } catch (PDOException $e) {
            $response['message'] = 'Error en cargar_afiliaciones: ' . $e->getMessage();
        }
        break;

    case 'obtener_afiliacion':
        if (!puede('ver_afiliaciones')) {
            $response['message'] = 'Acceso denegado.';
            break;
        }
        $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$id) {
            $response['message'] = 'ID de afiliación no válido.';
            break;
        }
        try {
            $sql_afiliacion = "SELECT a.*, s.nombre as nombre_sede, 
                                      p.nombre as nombre_plan, p.costo_mensual, p.miembros_max, p.costo_miembro_extra 
                               FROM afiliaciones a 
                               LEFT JOIN sedes s ON a.sede_id = s.id 
                               LEFT JOIN planes p ON a.plan_id = p.id 
                               WHERE a.id = ?";

            $params_afiliacion = [$id];

            if (!$es_superadmin) {
                if (empty($sedes_permitidas)) {
                    $response = ['success' => false, 'message' => 'No tiene sedes asignadas.'];
                    echo json_encode($response);
                    exit();
                }
                $placeholders = implode(',', array_fill(0, count($sedes_permitidas), '?'));
                $sql_afiliacion .= " AND a.sede_id IN ($placeholders)";
                $params_afiliacion = array_merge($params_afiliacion, $sedes_permitidas);
            }

            $stmt_titular = $conn->prepare($sql_afiliacion);
            $stmt_titular->execute($params_afiliacion);
            $afiliacion = $stmt_titular->fetch(PDO::FETCH_ASSOC);

            if ($afiliacion) {
                $stmt_miembros = $conn->prepare("SELECT * FROM afiliados_miembros WHERE afiliacion_id = ?");
                $stmt_miembros->execute([$id]);
                $afiliacion['miembros'] = $stmt_miembros->fetchAll(PDO::FETCH_ASSOC);

                // ***** INICIO DE LA CORRECCIÓN *****
                // Se cuenta al titular (1) + los miembros adicionales.
                $total_miembros_reales = 1 + count($afiliacion['miembros']);
                
                $costo_base = (float)($afiliacion['costo_mensual'] ?? 0);
                $miembros_max_base = (int)($afiliacion['miembros_max'] ?? 0);
                $costo_extra_por_miembro = (float)($afiliacion['costo_miembro_extra'] ?? 0);
                
                $miembros_extra = 0;
                // La comparación se hace con el total real de personas.
                if ($miembros_max_base > 0 && $total_miembros_reales > $miembros_max_base) {
                    $miembros_extra = $total_miembros_reales - $miembros_max_base;
                }

                $costo_total_extra = $miembros_extra * $costo_extra_por_miembro;
                $costo_total_mensual = $costo_base + $costo_total_extra;

                // Se envían los datos corregidos en la respuesta.
                $afiliacion['calculo'] = [
                    'costo_base' => $costo_base,
                    'miembros_incluidos' => $miembros_max_base,
                    'miembros_actuales' => $total_miembros_reales, // Se envía el total real
                    'miembros_extra' => $miembros_extra,
                    'costo_por_miembro_extra' => $costo_extra_por_miembro,
                    'costo_total_extra' => $costo_total_extra,
                    'costo_total_mensual' => $costo_total_mensual
                ];
                // ***** FIN DE LA CORRECCIÓN *****

                $response = ['success' => true, 'data' => $afiliacion];
            } else {
                $response['message'] = 'Afiliación no encontrada o no tiene permiso para verla.';
            }
        } catch (PDOException $e) {
            $response['message'] = 'Error en obtener_afiliacion: ' . $e->getMessage();
        }
        break;

    case 'guardar_afiliacion':
        if (!puede('gestionar_afiliaciones')) {
            $response['message'] = 'Acceso denegado para gestionar afiliaciones.';
            break;
        }
        
        $sede_id_seleccionada = filter_input(INPUT_POST, 'sede_id', FILTER_VALIDATE_INT);
        if (!$es_superadmin && !in_array($sede_id_seleccionada, $sedes_permitidas)) {
            $response['message'] = 'No tiene permisos para registrar o modificar afiliaciones en la sede seleccionada.';
            break;
        }

        try {
            $conn->beginTransaction();
            $afiliacion_id = filter_input(INPUT_POST, 'afiliacion_id', FILTER_VALIDATE_INT);
            
            $datos_nuevos = [
                'tipo_contrato' => trim($_POST['tipo_contrato']),
                'nombre_empresa' => $_POST['tipo_contrato'] === 'Empresarial' ? trim($_POST['nombre_empresa']) : null,
                'rif_empresa' => $_POST['tipo_contrato'] === 'Empresarial' ? trim($_POST['rif_empresa']) : null,
                'numero_contrato' => trim($_POST['numero_contrato']),
                'nombre_titular' => trim($_POST['nombre_titular']),
                'apellido_titular' => trim($_POST['apellido_titular']),
                'cedula_titular' => trim($_POST['cedula_titular']),
                'telefono_titular' => trim($_POST['telefono_titular']),
                'telefono_casa_titular' => trim($_POST['telefono_casa_titular']),
                'direccion_titular' => trim($_POST['direccion_titular']),
                'ciudad' => trim($_POST['ciudad']),
                'sede_id' => $sede_id_seleccionada,
                'plan_id' => filter_input(INPUT_POST, 'plan_id', FILTER_VALIDATE_INT),
                'fecha_inscripcion' => $_POST['fecha_inscripcion'],
                'estado' => $_POST['estado']
            ];

            if ($afiliacion_id) {
                if (!$es_superadmin) {
                    $stmt_check = $conn->prepare("SELECT COUNT(*) FROM afiliaciones WHERE id = ? AND sede_id IN (" . implode(',', array_fill(0, count($sedes_permitidas), '?')) . ")");
                    $check_params = array_merge([$afiliacion_id], $sedes_permitidas);
                    $stmt_check->execute($check_params);
                    if ($stmt_check->fetchColumn() == 0) {
                        throw new Exception("Intento de modificar una afiliación de una sede no autorizada.");
                    }
                }
                
                $sql = "UPDATE afiliaciones SET tipo_contrato=?, nombre_empresa=?, rif_empresa=?, numero_contrato=?, nombre_titular=?, apellido_titular=?, cedula_titular=?, telefono_titular=?, telefono_casa_titular=?, direccion_titular=?, ciudad=?, sede_id=?, plan_id=?, fecha_inscripcion=?, estado=? WHERE id=?";
                $params = array_values($datos_nuevos);
                $params[] = $afiliacion_id;
                $message = 'Afiliación actualizada correctamente.';
            } else {
                $sql = "INSERT INTO afiliaciones (tipo_contrato, nombre_empresa, rif_empresa, numero_contrato, nombre_titular, apellido_titular, cedula_titular, telefono_titular, telefono_casa_titular, direccion_titular, ciudad, sede_id, plan_id, fecha_inscripcion, estado, usuario_id, fecha_vencimiento) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $params = array_values($datos_nuevos);
                $params[] = $usuario_id;
                $params[] = $_POST['fecha_inscripcion'];
                $message = 'Afiliación creada correctamente.';
            }
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            if (!$afiliacion_id) { $afiliacion_id = $conn->lastInsertId(); }
            
            registrar_accion('guardar_afiliacion', "Se guardó la afiliación: N° Contrato " . $datos_nuevos['numero_contrato']);
            $conn->commit();
            $response = ['success' => true, 'message' => $message, 'id' => $afiliacion_id];
        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error en guardar_afiliacion: ' . $e->getMessage();
        }
        break;

    case 'gestionar_miembros':
        if (!puede('gestionar_afiliaciones')) {
            $response['message'] = 'Acceso denegado.';
            break;
        }
        $afiliacion_id = filter_input(INPUT_POST, 'afiliacion_id_miembros', FILTER_VALIDATE_INT);
        $miembros = $_POST['miembros'] ?? [];
        if (!$afiliacion_id) {
            $response['message'] = 'ID de afiliación no válido.';
            break;
        }
        try {
            $conn->beginTransaction();
            $conn->prepare("DELETE FROM afiliados_miembros WHERE afiliacion_id = ?")->execute([$afiliacion_id]);
            if (!empty($miembros)) {
                $stmt_miembro = $conn->prepare("INSERT INTO afiliados_miembros (afiliacion_id, nombre, apellido, cedula, fecha_nacimiento, parentesco) VALUES (?, ?, ?, ?, ?, ?)");
                foreach ($miembros as $miembro) {
                    if (!empty($miembro['nombre']) && !empty($miembro['apellido'])) {
                        $stmt_miembro->execute([ $afiliacion_id, trim($miembro['nombre']), trim($miembro['apellido']), trim($miembro['cedula']) ?: null, empty($miembro['fecha_nacimiento']) ? null : $miembro['fecha_nacimiento'], trim($miembro['parentesco']) ?: null ]);
                    }
                }
            }
            registrar_accion('gestionar_miembros', "Se actualizaron los miembros para la afiliación ID {$afiliacion_id}.");
            $conn->commit();
            $response = ['success' => true, 'message' => 'Lista de miembros actualizada.'];
        } catch(PDOException $e) {
            $conn->rollBack();
            $response['message'] = 'Error en gestionar_miembros: ' . $e->getMessage();
        }
        break;

    case 'verificar_clave_maestra':
        $clave = $_POST['clave_maestra'] ?? '';
        if (empty($clave)) {
            $response['message'] = 'Debe ingresar la clave.';
            break;
        }
        try {
            $stmt = $conn->prepare("SELECT clave_hash FROM claves_maestras WHERE estado = 'Activa'");
            $stmt->execute();
            $claves_activas = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
            $verificada = false;
            foreach ($claves_activas as $hash) {
                if (password_verify($clave, $hash)) {
                    $verificada = true;
                    break;
                }
            }
            if ($verificada) {
                $nombre_usuario = obtenerNombreUsuario($conn, $usuario_id);
                registrar_accion('uso_clave_maestra', "Clave maestra utilizada por {$nombre_usuario} en módulo de Afiliaciones.");
                $response = ['success' => true];
            } else {
                registrar_accion('intento_clave_maestra_fallido', "Intento fallido de clave maestra en Afiliaciones.");
                $response['message'] = 'Clave maestra incorrecta o inactiva.';
            }
        } catch (PDOException $e) {
            $response['message'] = 'Error en verificar_clave_maestra: ' . $e->getMessage();
        }
        break;

    case 'get_sedes':
        try {
            $sql_sedes = "SELECT id, nombre FROM sedes";
            $params_sedes = [];
            if (!$es_superadmin) {
                 if (empty($sedes_permitidas)) {
                    $response = ['success' => true, 'data' => []];
                    echo json_encode($response);
                    exit();
                }
                $placeholders = implode(',', array_fill(0, count($sedes_permitidas), '?'));
                $sql_sedes .= " WHERE id IN ($placeholders)";
                $params_sedes = $sedes_permitidas;
            }
            $sql_sedes .= " ORDER BY nombre ASC";
            
            $stmt = $conn->prepare($sql_sedes);
            $stmt->execute($params_sedes);
            $sedes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $sedes];
        } catch (PDOException $e) {
            $response['message'] = 'Error en get_sedes: ' . $e->getMessage();
        }
        break;

    case 'cargar_planes':
        if (!puede('ver_afiliaciones')) {
            $response['message'] = 'Acceso denegado.';
            break;
        }
        try {
            $stmt = $conn->query("SELECT * FROM planes WHERE estado = 'Activo' ORDER BY tipo_plan, costo_mensual");
            $planes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $planes];
        } catch (PDOException $e) {
            $response['message'] = 'Error en cargar_planes: ' . $e->getMessage();
        }
        break;

    case 'registrar_pago':
        // ***** INICIO DE LA MODIFICACIÓN *****
        if (!puede('gestionar_afiliaciones')) {
            $response['message'] = 'Acceso denegado para registrar pagos.';
            break;
        }
        
        // Se leen todos los campos del formulario
        $afiliacion_id = filter_input(INPUT_POST, 'afiliacion_id', FILTER_VALIDATE_INT);
        $cantidad_meses = filter_input(INPUT_POST, 'cantidad_meses', FILTER_VALIDATE_INT);
        $monto_pagado_usd = filter_input(INPUT_POST, 'monto_pagado', FILTER_VALIDATE_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        $fecha_pago = $_POST['fecha_pago'];
        $metodo_pago = trim($_POST['metodo_pago']);
        $referencia = trim($_POST['referencia']);
        
        // NUEVO: Se lee el monto en Bolívares. Si no se envía o está vacío, se trata como NULL.
        $monto_bs = filter_input(INPUT_POST, 'monto_bs', FILTER_VALIDATE_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        if ($monto_bs === false || $monto_bs === 0.0) {
            $monto_bs = null;
        }

        if (!$afiliacion_id || !$cantidad_meses || $cantidad_meses < 1 || $monto_pagado_usd === false) {
            $response['message'] = 'Datos de pago incompletos o inválidos.';
            break;
        }

        try {
            $conn->beginTransaction();
            
            $stmt_afil = $conn->prepare("SELECT fecha_inscripcion, fecha_vencimiento, plan_id, sede_id FROM afiliaciones WHERE id = ?");
            $stmt_afil->execute([$afiliacion_id]);
            $afiliacion = $stmt_afil->fetch(PDO::FETCH_ASSOC);
            
            if (!$afiliacion) { throw new Exception("La afiliación no existe."); }
            
            if (!$es_superadmin && !in_array($afiliacion['sede_id'], $sedes_permitidas)) {
                throw new Exception("No tiene permiso para registrar pagos para esta afiliación.");
            }

            $fecha_inscripcion_obj = new DateTime($afiliacion['fecha_inscripcion']);
            $dia_corte = $fecha_inscripcion_obj->format('d');
            
            $vencimiento_actual_obj = new DateTime($afiliacion['fecha_vencimiento'] ?? $afiliacion['fecha_inscripcion']);
            $vencimiento_calculado = clone $vencimiento_actual_obj;

            for ($i = 0; $i < $cantidad_meses; $i++) {
                // Lógica mejorada para sumar meses correctamente
                $vencimiento_calculado->modify('+1 month');
            }
            //Ajustar al día de corte original
            $vencimiento_calculado->setDate($vencimiento_calculado->format('Y'), $vencimiento_calculado->format('m'), $dia_corte);


            $periodo_desde = (clone $vencimiento_actual_obj)->format('Y-m-d');
            $periodo_hasta = $vencimiento_calculado->format('Y-m-d');

            // NUEVO: Se actualiza la consulta INSERT para incluir la nueva columna `monto_bs`
            $sql_pago = "INSERT INTO historial_pagos 
                         (afiliacion_id, plan_id, monto_pagado, monto_bs, fecha_pago, periodo_cubierto_desde, periodo_cubierto_hasta, metodo_pago, referencia, usuario_id) 
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt_pago = $conn->prepare($sql_pago);
            
            // NUEVO: Se añade $monto_bs a los parámetros que se ejecutan
            $stmt_pago->execute([
                $afiliacion_id, 
                $afiliacion['plan_id'], 
                $monto_pagado_usd, 
                $monto_bs, 
                $fecha_pago, 
                $periodo_desde, 
                $periodo_hasta, 
                $metodo_pago, 
                $referencia, 
                $usuario_id
            ]);

            $stmt_update = $conn->prepare("UPDATE afiliaciones SET fecha_vencimiento = ?, estado = 'Activo' WHERE id = ?");
            $stmt_update->execute([$periodo_hasta, $afiliacion_id]);

            $nombre_usuario = obtenerNombreUsuario($conn, $usuario_id);
            registrar_accion('registrar_pago', "Usuario {$nombre_usuario} registró pago de {$monto_pagado_usd} USD para afiliación ID {$afiliacion_id}. Nuevo vencimiento: {$periodo_hasta}.");
            
            $conn->commit();
            $response = ['success' => true, 'message' => '¡Pago registrado con éxito!'];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error en registrar_pago: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = "Acción desconocida: '{$action}'.";
        break;
}

echo json_encode($response);
?>